#include <stdio.h>
#include <string.h>
#include <time.h>

#define STORAGE_FILE "data.bin"
#define STORAGE_DIR  "/var/lib/resilience"

char storage_path[1024];

/*
 * FIPS 180-2 SHA-224/256/384/512 implementation
 * Last update: 02/02/2007
 * Issue date:  04/30/2005
 *
 * Copyright (C) 2013, Con Kolivas <kernel@kolivas.org>
 * Copyright (C) 2005, 2007 Olivier Gay <olivier.gay@a3.epfl.ch>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdint.h>

#define SHA256_DIGEST_SIZE ( 256 / 8)
#define SHA256_BLOCK_SIZE  ( 512 / 8)

#define SHFR(x, n)    (x >> n)
#define ROTR(x, n)   ((x >> n) | (x << ((sizeof(x) << 3) - n)))
#define CH(x, y, z)  ((x & y) ^ (~x & z))
#define MAJ(x, y, z) ((x & y) ^ (x & z) ^ (y & z))

#define SHA256_F1(x) (ROTR(x,  2) ^ ROTR(x, 13) ^ ROTR(x, 22))
#define SHA256_F2(x) (ROTR(x,  6) ^ ROTR(x, 11) ^ ROTR(x, 25))
#define SHA256_F3(x) (ROTR(x,  7) ^ ROTR(x, 18) ^ SHFR(x,  3))
#define SHA256_F4(x) (ROTR(x, 17) ^ ROTR(x, 19) ^ SHFR(x, 10))

typedef struct {
    unsigned int tot_len;
    unsigned int len;
    unsigned char block[2 * SHA256_BLOCK_SIZE];
    uint32_t h[8];
} sha256_ctx;

#define UNPACK32(x, str)                      \
{                                             \
    *((str) + 3) = (uint8_t) ((x)      );       \
    *((str) + 2) = (uint8_t) ((x) >>  8);       \
    *((str) + 1) = (uint8_t) ((x) >> 16);       \
    *((str) + 0) = (uint8_t) ((x) >> 24);       \
}

#define PACK32(str, x)                        \
{                                             \
    *(x) =   ((uint32_t) *((str) + 3)      )    \
           | ((uint32_t) *((str) + 2) <<  8)    \
           | ((uint32_t) *((str) + 1) << 16)    \
           | ((uint32_t) *((str) + 0) << 24);   \
}

#define SHA256_SCR(i)                         \
{                                             \
    w[i] =  SHA256_F4(w[i -  2]) + w[i -  7]  \
          + SHA256_F3(w[i - 15]) + w[i - 16]; \
}

uint32_t sha256_h0[8] =
            {0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a,
             0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19};

uint32_t sha256_k[64] =
            {0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5,
             0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
             0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3,
             0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
             0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc,
             0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
             0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7,
             0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
             0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13,
             0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
             0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3,
             0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
             0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5,
             0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
             0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208,
             0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2};

/* SHA-256 functions */

void sha256_transf(sha256_ctx *ctx, const unsigned char *message,
                   unsigned int block_nb)
{
    uint32_t w[64];
    uint32_t wv[8];
    uint32_t t1, t2;
    const unsigned char *sub_block;
    int i;

    int j;

    for (i = 0; i < (int) block_nb; i++) {
        sub_block = message + (i << 6);

        for (j = 0; j < 16; j++) {
            PACK32(&sub_block[j << 2], &w[j]);
        }

        for (j = 16; j < 64; j++) {
            SHA256_SCR(j);
        }

        for (j = 0; j < 8; j++) {
            wv[j] = ctx->h[j];
        }

        for (j = 0; j < 64; j++) {
            t1 = wv[7] + SHA256_F2(wv[4]) + CH(wv[4], wv[5], wv[6])
                + sha256_k[j] + w[j];
            t2 = SHA256_F1(wv[0]) + MAJ(wv[0], wv[1], wv[2]);
            wv[7] = wv[6];
            wv[6] = wv[5];
            wv[5] = wv[4];
            wv[4] = wv[3] + t1;
            wv[3] = wv[2];
            wv[2] = wv[1];
            wv[1] = wv[0];
            wv[0] = t1 + t2;
        }

        for (j = 0; j < 8; j++) {
            ctx->h[j] += wv[j];
        }
    }
}

void sha256_init(sha256_ctx *ctx)
{
    int i;
    for (i = 0; i < 8; i++) {
        ctx->h[i] = sha256_h0[i];
    }

    ctx->len = 0;
    ctx->tot_len = 0;
}

void sha256_update(sha256_ctx *ctx, const unsigned char *message,
                   unsigned int len)
{
    unsigned int block_nb;
    unsigned int new_len, rem_len, tmp_len;
    const unsigned char *shifted_message;

    tmp_len = SHA256_BLOCK_SIZE - ctx->len;
    rem_len = len < tmp_len ? len : tmp_len;

    memcpy(&ctx->block[ctx->len], message, rem_len);

    if (ctx->len + len < SHA256_BLOCK_SIZE) {
        ctx->len += len;
        return;
    }

    new_len = len - rem_len;
    block_nb = new_len / SHA256_BLOCK_SIZE;

    shifted_message = message + rem_len;

    sha256_transf(ctx, ctx->block, 1);
    sha256_transf(ctx, shifted_message, block_nb);

    rem_len = new_len % SHA256_BLOCK_SIZE;

    memcpy(ctx->block, &shifted_message[block_nb << 6],
           rem_len);

    ctx->len = rem_len;
    ctx->tot_len += (block_nb + 1) << 6;
}

void sha256_final(sha256_ctx *ctx, unsigned char *digest)
{
    unsigned int block_nb;
    unsigned int pm_len;
    unsigned int len_b;

    int i;

    block_nb = (1 + ((SHA256_BLOCK_SIZE - 9)
                     < (ctx->len % SHA256_BLOCK_SIZE)));

    len_b = (ctx->tot_len + ctx->len) << 3;
    pm_len = block_nb << 6;

    memset(ctx->block + ctx->len, 0, pm_len - ctx->len);
    ctx->block[ctx->len] = 0x80;
    UNPACK32(len_b, ctx->block + pm_len - 4);

    sha256_transf(ctx, ctx->block, block_nb);

    for (i = 0 ; i < 8; i++) {
        UNPACK32(ctx->h[i], &digest[i << 2]);
    }
}

void sha256(const unsigned char *message, unsigned int len, unsigned char *digest)
{
    sha256_ctx ctx;

    sha256_init(&ctx);
    sha256_update(&ctx, message, len);
    sha256_final(&ctx, digest);
}

unsigned char one_time_key_buf[SHA256_DIGEST_SIZE];

#define MAX_DG 508

unsigned char send_buf[MAX_DG];

unsigned char secret_block[32+4];

#define SCHEDULER_PERIOD 60

#define MAX_CAPACITY 16

#define TIMEOUT 300
#define PAYMENT_REQUEST_TIMEOUT 900
#define MAX_DEPTH 4
#define FEE_RATE 0.02

#define COMMIT 0
#define SEAL 1
#define FINALIZE 2
#define CANCEL 3
#define CLEANUP 4
#define SYNCING_MASK 0x06

struct remote {
    unsigned char username[32];
    unsigned char server_address[32];
    unsigned short port;
};

struct account {
    struct remote remote;
    unsigned char secret_key[32];
    unsigned int counter;
    long long balance;
    unsigned long long creditlimit;
    unsigned long long creditlimit_in;
    unsigned char ack_pending : 1;
    float width;
    unsigned long long direct_transfer;
    unsigned long long tax;
    unsigned char type;
    unsigned long long sync_in;
    unsigned long long sync_out;
    unsigned char ack : 1;
};

struct base_path {
    unsigned char identifier[32];
    unsigned long long amount;
    unsigned long long tax;
    signed char incoming;
    signed char outgoing;
    signed char counterpart;
    unsigned long long fee;
    unsigned char hops;
    long long time_value;
};

struct payment {
    struct base_path base;
    unsigned long long commit_penalty;
    unsigned long long finalize_out;
    unsigned char preimage[32];
    union {
        struct {
            unsigned char status      : 3;
            unsigned char commit_out  : 1;
            unsigned char ack_preimage: 1;
            unsigned char ack_sync    : 1;
            unsigned char synced      : 1;
        };
        unsigned char bits;
    };
};

struct receipt {
    unsigned char type;
    unsigned char identifier[32];
    unsigned char username[32];
    unsigned char server_address[32];
    long long amount;
    unsigned char status;
    long long timestamp;
};

struct storage {
    unsigned char username[32];
    unsigned char server_address[32];
    unsigned short port;
    unsigned char secret_key[32];
    unsigned int counter;
    float trust_index;
    struct account accounts[MAX_CAPACITY];
    unsigned short acc_bitmap;
    struct payment payments[MAX_CAPACITY];
    unsigned char num_payments;
    struct receipt receipts[MAX_CAPACITY];
    unsigned char receipts_head;
};

struct storage storage;

#include <unistd.h>

void save_storage(void) {
    char temp_path[1024+4];
    snprintf(temp_path, sizeof(temp_path), "%s.tmp", storage_path);
    FILE *fp = fopen(temp_path, "wb");
    if (!fp) return;
    
    if (fwrite(&storage, sizeof(struct storage), 1, fp) != 1) {
        fclose(fp);
        return;
    }
    if (fflush(fp) != 0 || fsync(fileno(fp)) != 0) {
        fclose(fp);
        return;
    }
    fclose(fp);
    rename(temp_path, storage_path);
}

signed char load_storage() {
    FILE *fp = fopen(storage_path, "rb");
    if (!fp) return -1;
    if (fread(&storage, sizeof(struct storage), 1, fp) != 1) {
        fclose(fp);
        return -1;
    }
    fclose(fp);
    return 0;
}

signed char lookup_account_idx(unsigned char *identifier) {
    unsigned short temp_bitmap = storage.acc_bitmap;
    signed char i = 0;
    while (temp_bitmap) {
        struct account *a = &storage.accounts[i];
        if ((temp_bitmap & 1) && strncmp(a->remote.username, identifier, 32) == 0 &&
            strncmp(a->remote.server_address, identifier + 32, 32) == 0) {
            return i;
        }
        i++;
        temp_bitmap >>= 1;
    };
    return -1;
}

struct auth {
    unsigned char secret_key[32];
    unsigned int counter_in;
    unsigned int counter_out;
};

struct session_auth {
    struct auth auth;
    unsigned char secret_block[32+4];
    unsigned char syncing : 1;
};

struct session_auth session_auth[MAX_CAPACITY];

struct counterpart {
    struct remote remote;
    struct auth auth;
    long long timeout;
};

struct counterpart counterparts[MAX_CAPACITY];
unsigned short cpt_bitmap;

struct pathfinding {
    struct base_path base;
    unsigned char depth;
    unsigned char in_or_out : 1;
    unsigned char commit : 1;
    unsigned char gas;
};

struct pathfinding paths[MAX_CAPACITY];
unsigned char num_paths;

unsigned long long direct_payment_pending[MAX_CAPACITY];
unsigned long long gas_buffer[MAX_CAPACITY];
unsigned long long tax_buffer[MAX_CAPACITY];

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/time.h>

#define HEADER_SIZE 5
#define MAX_RETRIES 8
#define BASE_DELAY 1
#define MAX_DELAY 16

#define ACCOUNT_NEW_SESSION 0
#define ACCOUNT_SESSION 2
#define COUNTERPART_SESSION 4
#define USER_SESSION 8

#define USER_SENDER (MAX_CAPACITY*2)
#define CPT_SENDER_BASE MAX_CAPACITY

#define TX_ENVELOPE (1+32+64+4+32)
#define REQ_ENVELOPE (1+4+32)

short endianess;

int sockfd;

time_t next_timer;

struct datagram {
    unsigned char payload[MAX_DG];
    unsigned short len;
    struct sockaddr_in addr;
};

struct send_rtx {
    signed char sender_idx;
    struct datagram send_queue[MAX_CAPACITY];
    unsigned char head;
    unsigned char tail;
    unsigned char size;
    time_t deadline;
    unsigned char retries;
};

struct send_rtx send_tasks[MAX_CAPACITY*2+1];
unsigned char num_active_sends = 0;

void remove_send_task(unsigned char idx) {
    struct send_rtx* t = &send_tasks[idx];
    t->head = (t->head + 1) % MAX_CAPACITY;
    t->size--;
    if (t->size == 0) {
        send_tasks[idx] = send_tasks[num_active_sends - 1];
        num_active_sends--;
        return;
    }
    struct datagram *dg = &t->send_queue[t->head];
    sendto(sockfd, dg->payload, dg->len, 0, (struct sockaddr *)&dg->addr, sizeof(struct sockaddr_in));
    t->retries = 0;
    time_t deadline = time(NULL) + BASE_DELAY;
    t->deadline = deadline;
    if (deadline < next_timer) {
        next_timer = deadline;
    }
}

void retransmit(time_t current_time) {
    for (unsigned char i = 0; i < num_active_sends; i++) {
        struct send_rtx *t = &send_tasks[i];
        unsigned char sender_idx = t->sender_idx;
        if (sender_idx < CPT_SENDER_BASE && session_auth[sender_idx].syncing) {
            continue;
        }
        if (t->deadline <= current_time) {
            struct datagram *dg = &t->send_queue[t->head];
            sendto(sockfd, dg->payload, dg->len, 0, (struct sockaddr *)&dg->addr, sizeof(struct sockaddr_in));
            t->retries++;
            if (t->retries >= MAX_RETRIES) {
                if (sender_idx < CPT_SENDER_BASE) {
                    session_auth[sender_idx].syncing = 1;
                    continue;
                }
                remove_send_task(i);
                i--;
                continue;
            }
            unsigned char delay = BASE_DELAY << t->retries;
            if (delay > MAX_DELAY) delay = MAX_DELAY;
            t->deadline = current_time + delay;
        }
        if (t->deadline < next_timer) {
            next_timer = t->deadline;
        }
    }
}

void enqueue_send(signed char sender_idx, unsigned short len, struct sockaddr_in *addr) {
    unsigned char idx = 0;
    while (idx < num_active_sends) {
        if (send_tasks[idx].sender_idx == sender_idx) {
            break;
        }
        idx++;
    }
    struct send_rtx* t = &send_tasks[idx];
    if (idx == num_active_sends) {
        memset(&t->head, 0, sizeof(struct send_rtx) - ((char*)&t->head-(char*)t));
        t->sender_idx = sender_idx;
        num_active_sends++;
    } else if (t->size >= MAX_CAPACITY) {
        return;
    }

    struct datagram *slot = &t->send_queue[t->tail];
    memcpy(slot->payload, send_buf, len);
    slot->len = len;
    slot->addr = *addr;
    t->tail = (t->tail + 1) % MAX_CAPACITY;
    t->size++;
    if (t->size == 1) {
        time_t deadline = time(NULL) + BASE_DELAY;
        t->deadline = deadline;
        if (deadline < next_timer) {
            next_timer = deadline;
        }
        sendto(sockfd, slot->payload, slot->len, 0, (struct sockaddr *)addr, sizeof(struct sockaddr_in));
    }
}

#include <netdb.h>

int create_sockaddr_in(const char *addr_str, unsigned short port, struct sockaddr_in *addr) {
    memset(addr, 0, sizeof(struct sockaddr_in));
    addr->sin_family = AF_INET;
    addr->sin_port = htons(port);
    if (inet_pton(AF_INET, addr_str, &addr->sin_addr) == 1) {
        return 0;
    } else {
        struct hostent *host = gethostbyname(addr_str);
        if (host == NULL) {
            return -1;
        }
        memcpy(&addr->sin_addr, host->h_addr, host->h_length);
        return 0;
    }
}

void remove_sender(signed char sender_idx) {
    for (unsigned char i = 0; i < num_active_sends; ++i) {
        if (send_tasks[i].sender_idx == sender_idx) {
            send_tasks[i] = send_tasks[--num_active_sends];
            return;
        }
    }
}

signed char lookup_cpt_idx(unsigned char *u, unsigned char *s) {
    unsigned short temp = cpt_bitmap;
    signed char i = 0;
    while (temp) {
        if ((temp & 1) &&
            strncmp(counterparts[i].remote.username, u, 32) == 0 &&
            strncmp(counterparts[i].remote.server_address, s, 32) == 0) {
            return i;
        }
        i++;
        temp >>= 1;
    }
    return -1;
}

signed char get_counterpart(struct base_path *base) {
    signed char idx = base->counterpart;
    unsigned char pay_id_buf[SHA256_DIGEST_SIZE];
    sha256(counterparts[idx].auth.secret_key, 32, pay_id_buf);
    if (memcmp(base->identifier, pay_id_buf, 32) == 0) {
        return idx;
    }
    return -1;
}

void remove_counterpart_by_idx(signed char idx) {
    remove_sender((signed char)(CPT_SENDER_BASE + idx));
    cpt_bitmap &= ~(1 << idx);
}

void remove_counterpart(struct base_path *base) {
    signed char idx = get_counterpart(base);
    if (idx != -1) {
        remove_counterpart_by_idx(idx);
    }
}

void write_receipt_internal(unsigned char *identifier, unsigned char *username, 
                           unsigned char *server_address, long long amount, 
                           unsigned char status, unsigned char type) {
    struct receipt *r = &storage.receipts[storage.receipts_head];
    storage.receipts_head = (storage.receipts_head + 1) % MAX_CAPACITY;

    r->type = type;

    if (identifier) {
        memcpy(r->identifier, identifier, 32);
    }
    
    memcpy(r->username, username, 32);

    memcpy(r->server_address, server_address, 32);
    r->amount = amount;
    r->status = status;
    r->timestamp = time(NULL);
    
    save_storage();
}

void write_receipt(struct base_path *base, unsigned char status) {
    long long amount = base->incoming == -1 ? -(long long)base->amount : (long long)base->amount;
    unsigned char *username;
    unsigned char *server_address;
    signed char cpt_idx = get_counterpart(base);
    if (cpt_idx != -1) {
        username = counterparts[cpt_idx].remote.username;
        server_address = counterparts[cpt_idx].remote.server_address;
    } else {
        unsigned char empty[32];
        memset(empty, 0, 32);
        username = empty;
        server_address = empty;
    }
    write_receipt_internal(base->identifier, username, server_address, amount, status, 0);
}

unsigned char is_counterpart(const struct base_path *b) {
    return b->counterpart != -1;
}

void remove_path_at(unsigned char idx, unsigned char receipt) {
    struct pathfinding *pf = &paths[idx];
    if (is_counterpart(&pf->base)) {
        if (receipt && pf->in_or_out) {
            write_receipt(&pf->base, 1);
        }
        remove_counterpart(&pf->base);
    }
    paths[idx] = paths[--num_paths];
}

void remove_payment_at(unsigned char pay_idx) {
    if (is_counterpart(&storage.payments[pay_idx].base)) {
        remove_counterpart(&storage.payments[pay_idx].base);
    }
    storage.payments[pay_idx] = storage.payments[--storage.num_payments];
}

struct pathfinding *get_pathfinding_by_idx(unsigned char idx) {
    struct pathfinding *pf = &paths[idx];
    if (pf->base.time_value < time(NULL)) {
        remove_path_at(idx, 1);
        return NULL;
    }
    return pf;
}

signed char get_pathfinding(unsigned char *payment_id) {
    signed char idx = 0;
    while (idx < num_paths) {
        struct pathfinding *pf = get_pathfinding_by_idx(idx);
        if (!pf) {
            continue;
        }
        if (memcmp(pf->base.identifier, payment_id, 32) == 0) {
            break;
        }
        idx++;
    }
    if (idx == num_paths) {
        return -1;
    }
    return idx;
}

void set_commit(struct pathfinding *pf) {
    pf->base.time_value = time(NULL) + TIMEOUT;
    pf->commit = 1;
}

unsigned long long fee(struct base_path *base, unsigned char in_or_out) {
    return ((unsigned long long)(base->hops + in_or_out)) * base->fee;
}
unsigned long long tax(struct base_path *base, unsigned char in_or_out) {
    return ((unsigned long long)base->hops + in_or_out) * base->tax;
}

unsigned long long fee_and_tax(struct base_path *base, unsigned char in_or_out) {
    return fee(base, in_or_out) + tax(base, in_or_out);
}
unsigned long long amount_fee_and_tax(struct base_path *base, unsigned char in_or_out) {
    return base->amount + fee_and_tax(base, in_or_out);
}

unsigned long long preview_bandwidth(signed char account_idx, unsigned char in_or_out) {
    struct account *acc = &storage.accounts[account_idx];

    long long bandwidth = in_or_out
        ? (long long)acc->creditlimit    - acc->balance
        : (long long)acc->creditlimit_in + acc->balance;

    for (unsigned char i = 0; i < storage.num_payments; i++) {
        struct payment *p = &storage.payments[i];
        if ((in_or_out ? p->base.incoming : p->base.outgoing) == account_idx) {
            bandwidth -= (long long)amount_fee_and_tax(&p->base, in_or_out);
        }
    }
    for (unsigned char i = 0; i < num_paths; i++) {
        struct pathfinding *pf = get_pathfinding_by_idx(i);
        if (!pf) {
            i--;
            continue;
        }
        if ((in_or_out ? pf->base.incoming : pf->base.outgoing) == account_idx && pf->commit) {
            bandwidth -= (long long)amount_fee_and_tax(&pf->base, in_or_out);
        }
    }
    return (unsigned long long)(bandwidth > 0 ? bandwidth : 0);
}

int verify_signature(unsigned char *data, unsigned short len, unsigned char *secret_key) {
    unsigned char hash[SHA256_DIGEST_SIZE];
    unsigned char signature[32];
    memcpy(signature, data + len - 32, 32);
    memcpy(data + len - 32, secret_key, 32);
    sha256(data, len, hash);
    return memcmp(hash, signature, 32) == 0;
}

void sign_data(unsigned short len, unsigned char *secret_key) {
    unsigned char signature[SHA256_DIGEST_SIZE];
    memcpy(send_buf + len, secret_key, 32);
    sha256(send_buf, len+32, signature);
    memcpy(send_buf + len, signature, 32);
}

void bswapn(unsigned char* dst, unsigned char* src, unsigned char n) {
    unsigned char fwd = 0;
    unsigned char rev = n - 1;
    unsigned char half = n/2;
    while (fwd < half) {
        dst[fwd] = src[rev];
        dst[rev] = src[fwd];
        fwd++;
        rev--;
    }
}

void ntohn(void* dst, unsigned char* src, unsigned char n, unsigned char size) {
    unsigned char *dst_c = (unsigned char*)dst;
    unsigned char padding = size-n;
    if (endianess) {
        bswapn(dst_c, src, n);
        dst_c = dst_c+n;
    } else {
        memcpy(dst_c+padding, src, n);
    }
    if (padding) {
        memset(dst_c, 0, padding);
    }
}

void htonn(void* src, unsigned char* dst, unsigned char n, unsigned char size) {
    unsigned char *src_c = (unsigned char*)src;
    if (endianess) {
        bswapn(dst, src_c, n);
    } else {
        memcpy(dst, src_c+size-n, n);
    }
}

void string_copy(unsigned char *dst, unsigned char *src, unsigned char size) {
    unsigned char i;
    for (i = 0; i < size - 1 && src[i]; i++) {
        dst[i] = src[i];
    }
    dst[i] = '\0';
}

void build_tx_and_sign(unsigned char session_type, struct remote *remote, unsigned char *secret_key, unsigned short len) {
    send_buf[0] = session_type;
    string_copy(send_buf+1, remote->username, 32);
    string_copy(send_buf+33, storage.username, 32);
    strcpy(send_buf+65, storage.server_address);
    sign_data(TX_ENVELOPE + len - 32, secret_key);
}

void remote_sign_and_send(unsigned char session_type, signed char sender_idx, struct remote *remote, unsigned char *secret_key, unsigned int counter, unsigned short len) {
    htonn(&counter, send_buf + 97, 4, sizeof(int));
    build_tx_and_sign(session_type, remote, secret_key, len);
    struct sockaddr_in addr;
    create_sockaddr_in(remote->server_address, remote->port, &addr);
    enqueue_send(sender_idx, TX_ENVELOPE + len, &addr);
}

void set_session_auth(unsigned char account_idx) {
    struct session_auth *sa = &session_auth[account_idx];
    htonn(&storage.accounts[account_idx].counter, sa->secret_block + 32, 4, sizeof(int));
    sha256(sa->secret_block, 36, sa->auth.secret_key);
    sa->auth.counter_in  = 0;
    sa->auth.counter_out = 0;
    sa->syncing = 0;
}

void new_session(unsigned char account_idx) {
    storage.accounts[account_idx].counter++;
    save_storage();
    set_session_auth(account_idx);
}

void set_remote(struct remote *remote, unsigned char *id, unsigned short port) {
    string_copy(remote->username, id, 32);
    string_copy(remote->server_address, id + 32, 31);
    remote->server_address[31] = '\0';
    remote->port = port;
}

unsigned short error_message(unsigned char *msg, unsigned short len) {
    send_buf[5] = 1;
    memcpy(send_buf + 6, msg, len);
    return len + 1;
}

unsigned short success_message(unsigned char *msg, unsigned short len) {
    send_buf[5] = 0;
    memcpy(send_buf + 6, msg, len);
    return len + 1;
}

unsigned short set_trust_index(unsigned char *data) {
    uint32_t bits;
    ntohn(&bits, data, 4, sizeof(uint32_t));
    float val = *((float*)&bits);
    storage.trust_index = val;
    save_storage();
    return success_message("Set trust index", 15);
}

unsigned short add_account(unsigned char *data) {
    unsigned char *id = data;
    unsigned short port;
    ntohn(&port, data + 64, 2, sizeof(short));
    unsigned char *secret_key = data + 66;

    if (lookup_account_idx(id) != -1) {
        return error_message("Peer already exists", 20);
    }
    signed char account_idx = 0;

    unsigned short temp_bitmap = storage.acc_bitmap;
    while (temp_bitmap) {
        if ((temp_bitmap & 1) == 0) {
            break;
        }
        account_idx++;
        temp_bitmap >>= 1;
    }

    if (account_idx == MAX_CAPACITY) {
        return error_message("Peer slots full", 16);
    }

    struct account *acc = &storage.accounts[account_idx];

    set_remote(&acc->remote, id, port);
    memcpy(acc->secret_key, secret_key, 32);
    memset(&acc->counter, 0, sizeof(struct account) - ((char*)&acc->counter - (char*)acc));
    storage.acc_bitmap |= (1 << account_idx);

    memcpy(session_auth[account_idx].secret_block, secret_key, 32);
    new_session(account_idx);
    remote_sign_and_send(ACCOUNT_NEW_SESSION, account_idx, &acc->remote, acc->secret_key, acc->counter, 0);

    return success_message("Add account", 12);
}

unsigned short remove_account(unsigned char *data) {
    unsigned char *id = data;
    signed char account_idx = lookup_account_idx(id);
    if (account_idx == -1) {
        return error_message("Peer does not exist", 20);
    }
    for (unsigned char i = 0; i < num_paths; ) {
        struct pathfinding *pf = &paths[i];
        if (pf->base.incoming == account_idx || pf->base.outgoing == account_idx) {
            remove_path_at(i, 0);
        } else {
            i++;
        }
    }
    for (unsigned char i = 0; i < storage.num_payments; ) {
        struct payment *p = &storage.payments[i];
        if (p->base.incoming == account_idx) {
            if (is_counterpart(&p->base)) {
                write_receipt(&p->base, 1);
            }
            p->base.incoming = -1;
        }
        if (p->base.outgoing == account_idx) {
            if (is_counterpart(&p->base)) {
                write_receipt(&p->base, p->status == CANCEL || p->status == COMMIT);
            }
            p->base.outgoing = -1;
        }
        if (p->base.incoming == -1 && p->base.outgoing == -1) {
            remove_payment_at(i);
        } else {
            i++;
        }
    }
    remove_sender(account_idx);
    storage.acc_bitmap &= ~(1 << account_idx);
    save_storage();
    return success_message("Remove account", 15);
}

#define ACCOUNT_FIND_PATH 0
#define ACCOUNT_PATH_RECURSE 1
#define ACCOUNT_PATH_FOUND 2
#define ACCOUNT_PREPARE_PATH 3
#define ACCOUNT_COMMIT_PAYMENT 4
#define ACCOUNT_SEAL_PAYMENT 5
#define ACCOUNT_FINALIZE_PAYMENT 6
#define ACCOUNT_CANCEL_PAYMENT 7
#define ACCOUNT_CLEANUP_PAYMENT 8
#define ACCOUNT_ACK_PREIMAGE 9
#define ACCOUNT_SYNC_PAYMENT 10
#define ACCOUNT_ACK_SYNC 11
#define ACCOUNT_VERIFY_COMMIT 12
#define ACCOUNT_ACK_COMMIT 13
#define ACCOUNT_DIRECT_PAYMENT 14
#define ACCOUNT_ACK_PAYMENT 15
#define ACCOUNT_SYNC_TRUSTLINE 16
#define ACCOUNT_ACK_TRUSTLINE 17
#define ACCOUNT_DIRECT_TRANSFER 18
#define ACCOUNT_ACK_TRANSFER 19

#define COUNTERPART_NEW_PAYMENT 0
#define COUNTERPART_START_PAYMENT 1
#define COUNTERPART_FIND_PATH 2
#define COUNTERPART_COMMIT_PAYMENT 3
#define COUNTERPART_SEAL_PAYMENT 4

#define USER_SET_TRUST_INDEX 0
#define USER_ADD_ACCOUNT 1
#define USER_REMOVE_ACCOUNT 2
#define USER_SET_TRUSTLINE 3
#define USER_NEW_PAYMENT 4
#define USER_RECEIVE_PAYMENT 5
#define USER_LOCAL_PAYMENT 6
#define USER_GET_TRUST_INDEX 7
#define USER_GET_NUM_ACCOUNTS 8
#define USER_GET_ACCOUNT 9
#define USER_GET_PAYMENT 10
#define USER_GET_RECEIPT 11

void account_sign_and_send(signed char account_idx, unsigned short len) {
    struct remote *remote = &storage.accounts[account_idx].remote;
    struct auth *auth = &session_auth[account_idx].auth;
    auth->counter_out++;
    remote_sign_and_send(ACCOUNT_SESSION, account_idx, remote, auth->secret_key, auth->counter_out, len);
}

void cpt_sign_and_send_idx(signed char cpt_idx, unsigned short len) {
    struct remote *remote = &counterparts[cpt_idx].remote;
    struct auth *auth = &counterparts[cpt_idx].auth;
    auth->counter_out++;
    remote_sign_and_send(COUNTERPART_SESSION, (signed char)(CPT_SENDER_BASE + cpt_idx), remote, auth->secret_key, auth->counter_out, len);
}

void cpt_send_cmd(signed char cpt_idx, unsigned char cmd) {
    send_buf[101] = cmd;
    cpt_sign_and_send_idx(cpt_idx, 1);
}

void account_send_32_bytes(signed char account_idx, unsigned char cmd, unsigned char* bytes) {
    send_buf[101] = cmd;
    memcpy(send_buf + 102, bytes, 32);
    account_sign_and_send(account_idx, 1 + 32);
}

void account_send_32_1(signed char account_idx, unsigned char cmd, unsigned char* bytes, unsigned char byte) {
    send_buf[101] = cmd;
    memcpy(send_buf + 102, bytes, 32);
    send_buf[134] = byte;
    account_sign_and_send(account_idx, 1 + 32 + 1);
}

void account_send_sync_payment(signed char account_idx, unsigned char* bytes, unsigned long long amount) {
    send_buf[101] = ACCOUNT_SYNC_PAYMENT;
    memcpy(send_buf + 102, bytes, 32);
    htonn(&amount, send_buf + 134, 8, sizeof(long long));
    account_sign_and_send(account_idx, 1 + 40);
}

void account_send_8(signed char account_idx, unsigned char cmd, unsigned long long val) {
    send_buf[101] = cmd;
    htonn(&val, send_buf + 102, 8, sizeof(long long));
    account_sign_and_send(account_idx, 1 + 8);
}

void account_send_16(signed char account_idx, unsigned char cmd, unsigned long long val1, unsigned long long val2) {
    send_buf[101] = cmd;
    htonn(&val1, send_buf + 102, 8, sizeof(long long));
    htonn(&val2, send_buf + 110, 8, sizeof(long long));
    account_sign_and_send(account_idx, 1 + 16);
}

void account_send_direct_payment(signed char account_idx) {
    send_buf[101] = ACCOUNT_DIRECT_PAYMENT;
    struct account *acc = &storage.accounts[account_idx];
    htonn(&acc->direct_transfer, send_buf + 102, 8, sizeof(long long));
    htonn(&acc->tax, send_buf + 110, 8, sizeof(long long));
    htonn(&acc->sync_out, send_buf + 118, 8, sizeof(long long));
    account_sign_and_send(account_idx, 1 + 24);
}

unsigned short set_trustline(unsigned char *data) {
    unsigned char *id = data;
    unsigned long long value;
    ntohn(&value, data + 64, 8, sizeof(long long));
    signed char account_idx = lookup_account_idx(id);
    if (account_idx == -1) {
        return error_message("Peer does not exist", 20);
    }
    struct account *acc = &storage.accounts[account_idx];
    acc->creditlimit = value;
    acc->ack_pending = 1;
    save_storage();
    account_send_8(account_idx, ACCOUNT_SYNC_TRUSTLINE, value);
    return success_message("Set trustline", 14);
}

unsigned char validate_payment_overflow(unsigned long long amount, unsigned long long fee, unsigned long long tax) {
    unsigned long long max_value = 0-1;
    if (amount > max_value >> 1 || (fee | tax) > max_value >> 10) {
        return 0;
    }
    return 1;
}

signed char set_counterpart(unsigned char *id, unsigned short port, unsigned char *cpt_secret_key) {
    signed char cpt_idx = lookup_cpt_idx(id, id + 32);
    if (cpt_idx != -1) {
        unsigned char pay_id_buf[SHA256_DIGEST_SIZE];
        sha256(counterparts[cpt_idx].auth.secret_key, 32, pay_id_buf);
        signed int pf_idx = get_pathfinding(pay_id_buf);
        if (pf_idx != -1) {
            struct pathfinding *pf = &paths[pf_idx];
            if (is_counterpart(&pf->base) && pf->in_or_out) {
                write_receipt(&pf->base, 1);
            }
            paths[pf_idx] = paths[--num_paths];
        }
        remove_counterpart_by_idx(cpt_idx);
    } else {
        cpt_idx = 0;
        long long now = time(NULL);
        unsigned short temp = cpt_bitmap;
        while (temp) {
            if ((temp & 1) == 0) break;
            if (counterparts[cpt_idx].timeout != 0 && counterparts[cpt_idx].timeout < now) {
                remove_counterpart_by_idx(cpt_idx);
                break;
            }
            cpt_idx++;
            temp >>= 1;
        }
        if (cpt_idx == MAX_CAPACITY) return -1;
    }

    cpt_bitmap |= (1 << cpt_idx);
    struct auth *auth = &counterparts[cpt_idx].auth;
    set_remote(&counterparts[cpt_idx].remote, id, port);
    memcpy(auth->secret_key, cpt_secret_key, 32);
    auth->counter_in = 0;
    auth->counter_out = 0;
    counterparts[cpt_idx].timeout = time(NULL) + PAYMENT_REQUEST_TIMEOUT;
    return cpt_idx;
}

void cpt_send_new_payment(signed char cpt_idx, unsigned char pf_idx) {
    struct pathfinding *pf = &paths[pf_idx];
    send_buf[101] = COUNTERPART_NEW_PAYMENT;
    unsigned char *data = send_buf+102;
    htonn(&pf->base.amount, data,  8, sizeof(long long));
    htonn(&pf->base.fee,    data + 8,  8, sizeof(long long));
    htonn(&pf->base.tax,    data + 16,  8, sizeof(long long));
    cpt_sign_and_send_idx(cpt_idx, 1+24);
}

signed char get_payment_idx(unsigned char *payment_id) {
    for (signed char i = 0; i < storage.num_payments; i++) {
        if (memcmp(storage.payments[i].base.identifier, payment_id, 32) == 0) {
            return i;
        }
    }
    return -1;
}

void add_pathfinding(unsigned char *payment_id, signed char cpt_idx, unsigned long long amount, unsigned long long tax, unsigned long long fee, unsigned char in_or_out) {
    struct pathfinding *pf = &paths[num_paths++];
    memcpy(pf->base.identifier, payment_id, 32);
    pf->base.amount = amount;
    pf->base.tax = tax;
    pf->base.incoming = -1;
    pf->base.outgoing = -1;
    pf->base.counterpart = cpt_idx;
    pf->base.fee = fee;
    pf->base.hops = 0;
    pf->base.time_value = time(NULL) + PAYMENT_REQUEST_TIMEOUT;
    pf->depth = 0;
    pf->in_or_out = in_or_out;
    pf->commit = 0;
    pf->gas = MAX_CAPACITY;
    counterparts[cpt_idx].timeout = 0;
}

unsigned short new_payment(unsigned char *data) {
    unsigned char *id = data;
    unsigned short port;
    ntohn(&port, data + 64, 2, sizeof(short));
    unsigned char *cpt_secret_key = data + 66;
    unsigned long long amount;
    ntohn(&amount, data + 98, 8, sizeof(long long));

    unsigned char pay_id_buf[SHA256_DIGEST_SIZE];
    sha256(cpt_secret_key, 32, pay_id_buf);

    if (get_pathfinding(pay_id_buf) != -1 || get_payment_idx(pay_id_buf) != -1) {
        return error_message("Payment already pending", 23);
    }

    signed char cpt_idx = set_counterpart(id, port, cpt_secret_key);
    if (cpt_idx < 0 || num_paths >= MAX_CAPACITY - storage.num_payments) {
        return error_message("Payment slots full", 19);
    }

    unsigned long long fee = (unsigned long long)((float)amount * FEE_RATE);
    unsigned long long tax = (unsigned long long)((float)amount * storage.trust_index);

    if (!validate_payment_overflow(amount, fee, tax) || amount == 0) {
        return error_message("Payment overflow", 17);
    }
    
    add_pathfinding(pay_id_buf, cpt_idx, amount, tax, fee, 1);
    cpt_send_new_payment(cpt_idx, num_paths-1);

    return success_message("New payment", 12);
}

unsigned short receive_payment(unsigned char *data) {
    unsigned char *id = data;
    unsigned short port;
    ntohn(&port, data + 64, 2, sizeof(short));
    unsigned char *cpt_secret_key = data + 66;

    signed char idx = set_counterpart(id, port, cpt_secret_key);
    if (idx < 0) return error_message("Payment slots full", 19);

    cpt_send_cmd(idx, COUNTERPART_START_PAYMENT);

    return success_message("Receive payment", 15);
}

void send_direct_payment(signed char account_idx, unsigned long long amount) {
    struct account *acc = &storage.accounts[account_idx];
    acc->sync_out++;
    acc->direct_transfer = amount;
    acc->tax = (unsigned long long)((float)amount * storage.trust_index);
    acc->type = 0;
    save_storage();
    account_send_direct_payment(account_idx);
}

unsigned short local_payment(unsigned char *data) {
    unsigned char *id = data;
    unsigned long long amount;
    ntohn(&amount, data + 64, 8, sizeof(long long));

    signed char account_idx = lookup_account_idx(id);
    if (account_idx == -1) {
        return error_message("Peer does not exist", 20);
    }
    struct account *acc = &storage.accounts[account_idx];

    if (direct_payment_pending[account_idx] > 0) {
        return error_message("Payment already queued", 22);
    }

    if (acc->direct_transfer > 0) {
        direct_payment_pending[account_idx] = amount;
        return success_message("Payment queued", 14);
    }

    send_direct_payment(account_idx, amount);
    return success_message("Local payment", 14);
}

unsigned short get_trust_index(unsigned char *data){
    send_buf[5] = 0;
    unsigned char *resp = send_buf + 6;
    htonn(&storage.trust_index, resp, 4, sizeof(float));
    return 5;
}

unsigned short get_num_accounts(unsigned char *data) {
    unsigned short temp_bitmap = storage.acc_bitmap;
    unsigned char count = 0;
    while (temp_bitmap) {
        if (temp_bitmap & 1) {
            count++;
        }
        temp_bitmap >>= 1;
    };
    send_buf[5] = 0;
    send_buf[6] = count;
    return 2;
}

unsigned short get_account(unsigned char *data) {
    unsigned char idx = data[0];
    unsigned short temp_bitmap = storage.acc_bitmap;
    unsigned char count = 0;
    unsigned char account_idx = 0;
    while (temp_bitmap) {
        if (temp_bitmap & 1) {
            if (count == idx) break;
            count++;
        }
        temp_bitmap >>= 1;
        account_idx++;
    }
    if (!temp_bitmap) {
        return error_message("Account not found", 18);
    }
    struct account *a = &storage.accounts[account_idx];

    send_buf[5] = 0;
    unsigned char *resp_data = send_buf+6;
    memcpy(resp_data, a->remote.username, 32);
    memcpy(resp_data+32, a->remote.server_address, 32);
    htonn(&a->balance, resp_data+64, 8, sizeof(long long));
    htonn(&a->creditlimit, resp_data+72, 8, sizeof(long long));
    htonn(&a->creditlimit_in, resp_data+80, 8, sizeof(long long));
    return 89;
}

unsigned short get_payment(unsigned char *data) {
    unsigned char idx = data[0];
    if (idx >= MAX_CAPACITY) {
        return error_message("Invalid index", 14);
    }
    
    send_buf[5] = 0;
    unsigned char *resp = send_buf + 6;
    
    if (!(cpt_bitmap & (1 << idx))) {
        resp[0] = 0;
        return 2;
    }
    
    struct counterpart *c = &counterparts[idx];
    unsigned char payment_id[32];
    sha256(c->auth.secret_key, 32, payment_id);
    
    struct base_path *base = NULL;
    
    signed char pf_idx = get_pathfinding(payment_id);
    if (pf_idx != -1) {
        struct pathfinding *pf = &paths[pf_idx];
        base = &pf->base;
        
        if (pf->depth == 0 && !pf->commit) {
            resp[0] = 1;
        } else {
            resp[0] = 2;
        }
        resp[105] = pf->in_or_out;
    } else {
        signed char pay_idx = get_payment_idx(payment_id);
        if (pay_idx != -1) {
            struct payment *p = &storage.payments[pay_idx];
            base = &p->base;
            resp[0] = 3;
            resp[105] = p->base.incoming == -1;
        }
    }
    
    if (!base) {
        resp[0] = 0;
        return 2;
    }
    
    memcpy(resp + 1, payment_id, 32);
    string_copy(resp + 33, c->remote.username, 32);
    string_copy(resp + 65, c->remote.server_address, 32);
    htonn(&base->amount, resp + 97, 8, sizeof(long long));
    htonn(&base->time_value, resp + 106, 8, sizeof(long long));
    
    return 115;
}

unsigned short get_receipt(unsigned char *data) {
    unsigned char idx = data[0];
    if (idx >= MAX_CAPACITY) {
        return error_message("Invalid index", 14);
    }
    struct receipt *r = &storage.receipts[idx];

    send_buf[5] = 0;
    unsigned char *resp = send_buf + 6;
    resp[0] = r->type;
    memcpy(resp + 1,        r->identifier,     32);
    memcpy(resp + 33,   r->username,       32);
    memcpy(resp + 65,   r->server_address, 32);
    htonn(&r->amount,    resp + 97,  8, sizeof(long long));
    htonn(&r->timestamp, resp + 105, 8, sizeof(long long));
    resp[113] = r->status;
    return 115;
}

void forward_find_path(unsigned char path_idx) {
    struct pathfinding *path = &paths[path_idx];
    if (path->depth >= MAX_DEPTH) return;

    unsigned char in_or_out = path->in_or_out;
    signed char from = in_or_out ? path->base.incoming : path->base.outgoing;

    unsigned int queries = path->gas;
    unsigned int gas = 1;
    for (unsigned char i = 0; i < path->depth; i++) {
        gas *= queries;
    }
    unsigned short temp_bitmap = storage.acc_bitmap;
    signed char i = 0;
    while (temp_bitmap && queries) {
        if (temp_bitmap & 1 && i != from && preview_bandwidth(i, !in_or_out) >= path->base.amount) {
            send_buf[101] = ACCOUNT_FIND_PATH;
            unsigned char *data = send_buf+102;
            memcpy(data, path->base.identifier, 32);
            htonn(&path->base.amount, data+32, 8, sizeof(long long));
            htonn(&path->base.tax, data+40, 8, sizeof(long long));
            data[48] = in_or_out;
            htonn(&path->base.fee, data+49, 8, sizeof(long long));
            data[57] = path->base.hops;
            data[58] = path->gas;
            account_sign_and_send(i, 1 + 59);
            
            gas_buffer[i] += gas;
            queries--;
        }
        i++;
        temp_bitmap >>= 1;
    }
    path->depth++;
}

unsigned char is_path_found(struct pathfinding *pf) {
    return pf->base.incoming != -1 && (pf->base.outgoing != -1 || is_counterpart(&pf->base));
}

unsigned char buyer_can_prepare(struct pathfinding *pf) {
    return preview_bandwidth(pf->base.outgoing, 0) >= amount_fee_and_tax(&pf->base, 0) && pf->base.hops <= pf->depth * 2 + 1;
}

void find_path(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;
    unsigned long long amount;
    ntohn(&amount, args + 32, 8, sizeof(long long));
    unsigned long long tax;
    ntohn(&tax, args + 40, 8, sizeof(long long));
    unsigned char in_or_out = args[48];
    unsigned long long fee;
    ntohn(&fee, args + 49, 8, sizeof(long long));
    unsigned char hops = args[57];
    unsigned char gas = args[58];
    if (preview_bandwidth(account_idx, in_or_out) < amount) {
        return;
    }

    signed char pf_idx = get_pathfinding(payment_id);

    if (pf_idx == -1) {
        if (amount == 0 || (float)fee / (float)amount < FEE_RATE / 2) {
            return;
        }

        float tax_rate = (float)tax / (float)amount;
        if(tax_rate < storage.trust_index / 2) {
            return;
        }
        if (!validate_payment_overflow(amount, fee, tax)) {
            return;
        }
        if (num_paths >= (MAX_CAPACITY - storage.num_payments) / 2) {
            return;
        }
        pf_idx = num_paths++;
        struct pathfinding *pf = &paths[pf_idx];
        memcpy(pf->base.identifier, payment_id, 32);
        pf->base.amount = amount;
        pf->base.counterpart = -1;
        pf->in_or_out = in_or_out;
        pf->base.fee = fee;
        pf->base.tax = tax;
        pf->base.time_value = time(NULL) + TIMEOUT;
        pf->depth = 0;
        pf->commit = 0;
        pf->gas = gas;
        if (in_or_out) {
            pf->base.incoming = account_idx;
            pf->base.outgoing = -1;
            pf->base.hops = 0;
        } else {
            pf->base.incoming = -1;
            pf->base.outgoing = account_idx;
            pf->base.hops = hops+1;
        }
        account_send_32_1(account_idx, ACCOUNT_PATH_RECURSE, payment_id, 0);
        return;
    }
    struct pathfinding *pf = &paths[pf_idx];

    if (pf->commit || is_path_found(pf)) {
        return;
    }

    if (in_or_out != pf->in_or_out) {

        if (amount != pf->base.amount ||
            fee != pf->base.fee ||
            tax != pf->base.tax) {
            return;
        }

        if (in_or_out) {
            pf->base.incoming = account_idx;
        } else {
            pf->base.outgoing = account_idx;
            pf->base.hops = hops + 1;
        }
        if (is_counterpart(&pf->base) && pf->in_or_out) {
            if (buyer_can_prepare(pf)) {
                set_commit(pf);
                account_send_32_bytes(account_idx, ACCOUNT_PREPARE_PATH, payment_id);
            }
            return;
        }
        account_send_32_1(pf->base.incoming, ACCOUNT_PATH_FOUND, payment_id, pf->base.hops);
        return;
    }
    if ((in_or_out ? pf->base.incoming : pf->base.outgoing) == account_idx) {
        forward_find_path(pf_idx);
    }
}

void path_recurse(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;
    unsigned char depth = args[32];

    signed char pf_idx = get_pathfinding(payment_id);
    if (pf_idx == -1) {
        return;
    }
    struct pathfinding *pf = &paths[pf_idx];

    if (pf->commit || is_path_found(pf) || depth + 1 != pf->depth) {
        return;
    }

    if (is_counterpart(&pf->base)) {
        signed char cpt_idx = get_counterpart(&pf->base);
        if (cpt_idx >= 0) cpt_send_cmd(cpt_idx, COUNTERPART_FIND_PATH);
        return;
    }

    signed char prev_idx = pf->in_or_out ? pf->base.incoming : pf->base.outgoing;

    account_send_32_1(prev_idx, ACCOUNT_PATH_RECURSE, payment_id, pf->depth);
}

void path_found(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;
    unsigned char hops = args[32];

    signed char pf_idx = get_pathfinding(payment_id);
    if (pf_idx == -1) {
        return;
    }
    struct pathfinding *pf = &paths[pf_idx];

    if (is_path_found(pf) || pf->commit || !pf->in_or_out) {
        return;
    }

    pf->base.outgoing = account_idx;
    pf->base.hops = hops + 1;

    if (is_counterpart(&pf->base)) {
        if (buyer_can_prepare(pf)) {
            set_commit(pf);
            account_send_32_bytes(account_idx, ACCOUNT_PREPARE_PATH, payment_id);
        }
        return;
    }

    account_send_32_1(pf->base.incoming, ACCOUNT_PATH_FOUND, payment_id, pf->base.hops);
}

void prepare_path(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;

    signed char pf_idx = get_pathfinding(payment_id);
    if (pf_idx == -1) {
        return;
    }
    struct pathfinding *pf = &paths[pf_idx];

    if (pf->commit || preview_bandwidth(account_idx, 1) < amount_fee_and_tax(&pf->base, 1)) {
        return;
    }

    if (is_counterpart(&pf->base)) {
        if (pf->in_or_out) {
            return;
        }
        pf->base.incoming = account_idx;
        set_commit(pf);
        signed char cpt_idx = get_counterpart(&pf->base);
        if (cpt_idx >= 0) cpt_send_cmd(cpt_idx, COUNTERPART_COMMIT_PAYMENT);
        return;
    }

    if (pf->base.outgoing == -1 || preview_bandwidth(pf->base.outgoing, 0) < amount_fee_and_tax(&pf->base, 0)) {
        return;
    }

    pf->base.incoming = account_idx;
    set_commit(pf);

    account_send_32_bytes(pf->base.outgoing, ACCOUNT_PREPARE_PATH, payment_id);
}

void commit_payment(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;

    signed char pf_idx = get_pathfinding(payment_id);
    if (pf_idx == -1) {
        return;
    }
    struct pathfinding *pf = &paths[pf_idx];

    if (!pf->commit || account_idx != pf->base.incoming) {
        return;
    }

    struct payment *payment = &storage.payments[storage.num_payments];
    storage.num_payments++;

    memcpy(&payment->base, &pf->base, (char*)&pf->base.time_value - (char*)&pf->base);
    payment->base.time_value = time(NULL);

    payment->bits = 0;

    if (is_counterpart(&pf->base)) {
        signed char cpt_idx = get_counterpart(&pf->base);
        if (cpt_idx < 0) { paths[pf_idx] = paths[--num_paths]; return; }
        memcpy(payment->preimage, counterparts[cpt_idx].auth.secret_key, 32);
        save_storage();
        paths[pf_idx] = paths[--num_paths];
        cpt_send_cmd(cpt_idx, COUNTERPART_SEAL_PAYMENT);
        return;
    }
    paths[pf_idx] = paths[--num_paths];
    save_storage();

    account_send_32_bytes(pf->base.outgoing, ACCOUNT_COMMIT_PAYMENT, payment_id);
}

unsigned long long penalty_ticker(struct payment *payment) {
    unsigned long long ticker = 1;
    long long elapsed = time(NULL) - payment->base.time_value;
    if (elapsed > 0) {
        ticker += (unsigned long long)elapsed;
    }
    return ticker > payment->base.amount ? payment->base.amount : ticker;
}

void seal_payment(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;

    signed char pay_idx = get_payment_idx(payment_id);
    if (pay_idx == -1) {
        return;
    }
    struct payment *payment = &storage.payments[pay_idx];

    if (payment->status != COMMIT || payment->base.incoming != account_idx) {
        return;
    }

    unsigned long long ticker = penalty_ticker(payment);
    payment->commit_penalty = ticker;

    if (ticker >= payment->base.amount) {
        payment->status = CLEANUP;
        save_storage();
        account_send_32_bytes(account_idx, ACCOUNT_CLEANUP_PAYMENT, payment_id);

        if (payment->base.outgoing != -1) {
            payment->finalize_out = amount_fee_and_tax(&payment->base, 0);
            save_storage();
            account_send_32_bytes(payment->base.outgoing, ACCOUNT_CLEANUP_PAYMENT, payment_id);
        }
        return;
    }

    if (is_counterpart(&payment->base)) {
        payment->status = FINALIZE;
        save_storage();
        account_send_32_bytes(account_idx, ACCOUNT_FINALIZE_PAYMENT, payment->preimage);
        return;
    }
    payment->status = SEAL;
    save_storage();
    if (payment->base.outgoing != -1) {
        account_send_32_bytes(payment->base.outgoing, ACCOUNT_SEAL_PAYMENT, payment_id);
    }
}

unsigned long long seal_penalty(struct payment *payment, unsigned long long ticker) {
    return ticker > payment->commit_penalty ? ticker - payment->commit_penalty : 0;
}

unsigned long long finalize_out(struct payment *payment) {
    unsigned long long ticker = penalty_ticker(payment);
    unsigned long long amount = payment->base.amount;
    unsigned long long amount_fin = amount - seal_penalty(payment, ticker);
    unsigned long long tax_fin = (unsigned long long)((double)tax(&payment->base, 0) * (double)amount_fin / (double)amount);
    unsigned long long fee_fin = (unsigned long long)((double)fee(&payment->base, 0) * (double)ticker / (double)amount);
    return amount_fin + tax_fin + fee_fin;
}

void send_ack_preimage(signed char sender_idx, unsigned char *payment_id) {
    account_send_32_bytes(sender_idx, ACCOUNT_ACK_PREIMAGE, payment_id);
}

void finalize_payment(signed char account_idx, unsigned char *args) {
    unsigned char *preimage_arg = args;
    unsigned char payment_id[32];
    sha256(preimage_arg, 32, payment_id);

    signed char pay_idx = get_payment_idx(payment_id);
    if (pay_idx == -1) {
        send_ack_preimage(account_idx, payment_id);
        return;
    }
    struct payment *payment = &storage.payments[pay_idx];

    if (payment->status == FINALIZE || payment->status == CLEANUP) {
        send_ack_preimage(account_idx, payment_id);
        return;
    }

    if (payment->status != SEAL || payment->base.outgoing != account_idx) {
        return;
    }

    payment->finalize_out = finalize_out(payment);
    payment->status = FINALIZE;
    memcpy(payment->preimage, preimage_arg, 32);
    save_storage();

    account_send_sync_payment(account_idx, payment_id, payment->finalize_out);
    send_ack_preimage(account_idx, payment_id);

    if (payment->base.incoming != -1) {
        account_send_32_bytes(payment->base.incoming, ACCOUNT_FINALIZE_PAYMENT, preimage_arg);
    }
}

unsigned long long cancel_out(struct payment *payment) {
    unsigned long long ticker = penalty_ticker(payment);
    unsigned long long total = amount_fee_and_tax(&payment->base, 0);
    if (ticker >= payment->base.amount) {
        return total;
    } else {
        return total * ticker / payment->base.amount;
    }
}

void cancel_payment(signed char account_idx, unsigned char *args) {
    unsigned char *preimage_arg = args;

    unsigned char payment_id[32];
    sha256(preimage_arg, 32, payment_id);

    signed char pay_idx = get_payment_idx(payment_id);
    if (pay_idx == -1) {
        send_ack_preimage(account_idx, payment_id);
        return;
    }
    struct payment *payment = &storage.payments[pay_idx];

    unsigned char status = payment->status;
    if (status == CANCEL || status == CLEANUP) {
        send_ack_preimage(account_idx, payment_id);
        return;
    }

    if (status != COMMIT || payment->base.incoming != account_idx) {
        return;
    }

    memcpy(payment->preimage, preimage_arg, 32);
    payment->status = CANCEL;

    if (payment->base.outgoing != -1) {
        payment->finalize_out = cancel_out(payment);
    }

    save_storage();

    if (payment->base.outgoing != -1) {
        account_send_32_bytes(payment->base.outgoing, ACCOUNT_CANCEL_PAYMENT, payment->preimage);
    }
    send_ack_preimage(account_idx, payment_id);
}

unsigned long long cleanup_out(struct payment *payment) {
    return payment->base.amount - seal_penalty(payment, payment->base.amount) + fee_and_tax(&payment->base, 0);
}

unsigned char counterpart_in(struct payment *p) {
    return is_counterpart(&p->base) && p->base.incoming == -1;
}

void cleanup_payment(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;

    signed char pay_idx = get_payment_idx(payment_id);
    if (pay_idx == -1) {
        return;
    }
    struct payment *payment = &storage.payments[pay_idx];

    if ((payment->status & SYNCING_MASK) || penalty_ticker(payment) < payment->base.amount) {
        return;
    }

    if (payment->status == COMMIT) {
        if (counterpart_in(payment)) return;
        payment->commit_penalty = payment->base.amount;
    }

    payment->status = CLEANUP;

    if (payment->base.outgoing != -1) {
        payment->finalize_out = cleanup_out(payment);
        save_storage();
        if (payment->base.outgoing != account_idx) {
            account_send_32_bytes(payment->base.outgoing, ACCOUNT_CLEANUP_PAYMENT, payment_id);
        }
    }

    save_storage();

    if (payment->base.incoming != -1 && payment->base.incoming != account_idx) {
        account_send_32_bytes(payment->base.incoming, ACCOUNT_CLEANUP_PAYMENT, payment_id);
    }
}

unsigned char synchronized_in(struct payment *payment) {
    unsigned char ack_preimage = payment->status != FINALIZE || payment->ack_preimage;
    return payment->synced && ack_preimage || counterpart_in(payment);
}

unsigned char synchronized_out(struct payment *payment) {
    unsigned char ack_preimage = payment->status != CANCEL || payment->ack_preimage;
    unsigned char commit_out = payment->status == FINALIZE || payment->commit_out;
    return payment->ack_sync && ack_preimage && commit_out || is_counterpart(&payment->base) && payment->base.outgoing == -1;
}

void ack_preimage(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;

    signed char pay_idx = get_payment_idx(payment_id);
    if (pay_idx == -1) {
        return;
    }
    struct payment *payment = &storage.payments[pay_idx];

    unsigned char status = payment->status;
    if (payment->ack_preimage ||
        !((status == FINALIZE && payment->base.incoming == account_idx) ||
          (status == CANCEL && payment->base.outgoing == account_idx))) {
        return;
    }

    payment->ack_preimage = 1;

    if (synchronized_in(payment)) {
        payment->base.incoming = -1;
    }
    if (synchronized_out(payment)) {
        payment->base.outgoing = -1;
    }

    if (payment->base.incoming == -1 && payment->base.outgoing == -1) {
        remove_payment_at(pay_idx);
    }
    save_storage();
}

long long preview_balance(signed char account_idx) {
    struct account *acc = &storage.accounts[account_idx];
    long long bal = acc->balance;
    for (unsigned char i = 0; i < storage.num_payments; i++) {
        struct payment *p = &storage.payments[i];
        if (p->base.incoming == account_idx) {
            bal -= (long long)amount_fee_and_tax(&p->base, 1);
        }
    }
    for (unsigned char i = 0; i < num_paths; i++) {
        struct pathfinding *pf = get_pathfinding_by_idx(i);
        if (!pf) {
            i--;
            continue;
        }
        if (pf->base.incoming == account_idx && pf->commit) {
            bal -= (long long)amount_fee_and_tax(&pf->base, 1);
        }
    }
    return bal;
}

unsigned long long calculate_tax(struct payment *payment) {
    return payment->base.tax * (payment->base.amount - seal_penalty(payment, penalty_ticker(payment)))/payment->base.amount;
}

void redistribute(signed char from_idx, unsigned long long tax, float tax_rate) {
    if (tax == 0) return;
    struct { signed char idx; unsigned long long val; } positive_balances[MAX_CAPACITY];
    unsigned char num_positive = 0;
    float total_width = 0.0;
    unsigned short temp_bitmap = storage.acc_bitmap;
    signed char i = 0;
    while (temp_bitmap) {
        if (temp_bitmap & 1 && i != from_idx) {
            long long preview = preview_balance(i);
            if (preview > 0) {
                positive_balances[num_positive].idx = i;
                positive_balances[num_positive].val = (unsigned long long)preview;
                total_width += storage.accounts[i].width;
                num_positive++;
            }
        }
        i++;
        temp_bitmap >>= 1;
    }
    if (total_width == 0.0) return;
    if (total_width < tax_rate) {
        tax = (unsigned long long)((float)tax * (total_width / tax_rate));
    }
    for (unsigned char j = 0; j < num_positive; j++) {
        signed char idx = positive_balances[j].idx;
        struct account *acc = &storage.accounts[idx];
        float proportion = acc->width / total_width;
        unsigned long long dist = (unsigned long long)((float)tax * proportion);
        if (dist == 0) continue;
        if (dist > positive_balances[j].val) dist = positive_balances[j].val;
        tax_buffer[idx] += dist;
        if (acc->direct_transfer == 0) {
            acc->sync_out++;
            acc->direct_transfer = tax_buffer[idx];
            acc->type = 2;
            save_storage();

            send_buf[101] = ACCOUNT_DIRECT_TRANSFER;
            htonn(&acc->direct_transfer, send_buf + 102, 8, sizeof(long long));
            htonn(&acc->sync_out, send_buf + 110, 8, sizeof(long long));
            send_buf[118] = 1;
            account_sign_and_send(idx, 1 + 17);

            tax_buffer[idx] = 0;
        }
    }
}

float get_tax_rate(struct payment *p) {
    if (p->base.tax == 0) {
        return 0;
    }
    return (float)p->base.tax / (float)(p->base.amount + p->base.tax);
}

void update_balance(signed char account_idx, long long amount, float tax_rate) {
    struct account *acc = &storage.accounts[account_idx];
    long long old_balance = acc->balance;
    long long new_balance = old_balance + amount;
    acc->balance = new_balance;
    if ((amount > 0 && new_balance > 0) || (amount < 0 && new_balance < 0)) {
        if ((old_balance > 0) == (new_balance > 0)) {
            float weight = (float)old_balance / (float)new_balance;
            acc->width = weight * acc->width + (1 - weight) * tax_rate;
        } else {
            acc->width = tax_rate;
        }
    }
}

void sync_payment(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;

    unsigned long long amount;
    ntohn(&amount, args + 32, 8, sizeof(long long));

    signed char pay_idx = get_payment_idx(payment_id);
    if (pay_idx == -1) {
        account_send_32_bytes(account_idx, ACCOUNT_ACK_SYNC, payment_id);
        return;
    }

    struct payment *payment = &storage.payments[pay_idx];

    if (!(payment->status & SYNCING_MASK) || (payment->base.incoming != -1 && payment->base.incoming != account_idx) || amount > amount_fee_and_tax(&payment->base, 1)) {
        return;
    }

    if (payment->base.incoming == account_idx && !payment->synced) {
        float tax_rate = get_tax_rate(payment);
        update_balance(payment->base.incoming, (long long)amount, tax_rate);

        payment->synced = 1;

        if (is_counterpart(&payment->base)) {
            write_receipt(&payment->base, payment->status == CANCEL);
        }
        if (synchronized_in(payment)) {
            payment->base.incoming = -1;
        }
        save_storage();
        redistribute(account_idx, calculate_tax(payment), tax_rate);
    }

    if (payment->base.incoming == -1 && payment->base.outgoing == -1) {
        remove_payment_at(pay_idx);
        save_storage();
    }

    account_send_32_bytes(account_idx, ACCOUNT_ACK_SYNC, payment_id);
}

void ack_sync(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;

    signed char pay_idx = get_payment_idx(payment_id);
    if (pay_idx == -1) {
        return;
    }

    struct payment *payment = &storage.payments[pay_idx];

    if (!(payment->status & SYNCING_MASK) || payment->ack_sync || payment->base.outgoing != account_idx) {
        return;
    }
    payment->ack_sync = 1;

    update_balance(payment->base.outgoing, -(long long)payment->finalize_out, get_tax_rate(payment));

    if (is_counterpart(&payment->base)) {
        write_receipt(&payment->base, payment->status == CANCEL);
    }
    if (synchronized_out(payment)) {
        payment->base.outgoing = -1;
    }
    if (payment->base.incoming == -1 && payment->base.outgoing == -1) {
        remove_payment_at(pay_idx);
    }
    save_storage();
}

void verify_commit(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;

    signed char pay_idx = get_payment_idx(payment_id);
    if (pay_idx == -1) {
        account_send_32_1(account_idx, ACCOUNT_ACK_COMMIT, payment_id, 1);
        return;
    }

    struct payment *payment = &storage.payments[pay_idx];

    if (payment->base.incoming != account_idx) {
        account_send_32_1(account_idx, ACCOUNT_ACK_COMMIT, payment_id, 1);
        return;
    }
    account_send_32_1(account_idx, ACCOUNT_ACK_COMMIT, payment_id, 0);
}

void ack_commit(signed char account_idx, unsigned char *args) {
    unsigned char *payment_id = args;
    unsigned char status = args[32];

    signed char pay_idx = get_payment_idx(payment_id);
    if (pay_idx == -1) {
        return;
    }

    struct payment *payment = &storage.payments[pay_idx];

    if (payment->commit_out || payment->base.outgoing != account_idx) {
        return;
    }

    if (status == 0) {
        payment->commit_out = 1;
        save_storage();
        if (payment->status & SYNCING_MASK) {
            account_send_sync_payment(payment->base.outgoing, payment->base.identifier, payment->finalize_out);
        }
    } else {
        payment->base.outgoing = -1;
        if (payment->base.incoming == -1) {
            if (is_counterpart(&payment->base)) {
                write_receipt(&payment->base, 1);
            }
            remove_payment_at(pay_idx);
            save_storage();
        }
    }
}

void direct_payment(signed char account_idx, unsigned char *args) {
    unsigned long long amount;
    ntohn(&amount, args, 8, sizeof(long long));
    unsigned long long tax;
    ntohn(&tax, args + 8, 8, sizeof(long long));
    unsigned long long counter;
    ntohn(&counter, args + 16, 8, sizeof(long long));
    
    struct account *acc = &storage.accounts[account_idx];
    
    if (counter < acc->sync_in) {
        return;
    }

    send_buf[101] = ACCOUNT_ACK_PAYMENT;
    htonn(&counter, send_buf + 102, 8, sizeof(long long));

    if (counter == acc->sync_in) {
        send_buf[110] = acc->ack;
        account_sign_and_send(account_idx, 1 + 9);
        return;
    }
    
    unsigned long long total = amount + tax;
    
    if (total == 0) {
        return;
    }
    float tax_rate = (float)tax / (float)total;

    unsigned char status;
    
    if (preview_bandwidth(account_idx, 1) < total) {
        status = 1;
    } else {
        status = 0;
        
        update_balance(account_idx, (long long)total, tax_rate);
    }
    
    acc->ack = status;
    acc->sync_in = counter;
    save_storage();
    
    write_receipt_internal(NULL, acc->remote.username, acc->remote.server_address, 
                          (long long)total, status, 1);

    send_buf[110] = status;
    account_sign_and_send(account_idx, 1 + 9);
    
    if (status == 0 && tax > 0) {
        redistribute(account_idx, tax, tax_rate);
    }
}

void ack_payment(signed char account_idx, unsigned char *args) {
    unsigned long long counter;
    ntohn(&counter, args, 8, sizeof(long long));
    unsigned char status = args[8];
    
    struct account *acc = &storage.accounts[account_idx];
    
    if (acc->direct_transfer > 0 && acc->type == 0 && counter == acc->sync_out) {
        long long amount = (long long)(acc->direct_transfer + acc->tax);
        
        if (status == 0) {
            float tax_rate = (float)acc->tax / (float)(acc->direct_transfer + acc->tax);
            update_balance(account_idx, -amount, tax_rate);
        }
        
        write_receipt_internal(NULL, acc->remote.username, acc->remote.server_address, 
                              -amount, status, 1);
        
        acc->direct_transfer = 0;
        if (direct_payment_pending[account_idx] > 0) {
            unsigned long long next = direct_payment_pending[account_idx];
            direct_payment_pending[account_idx] = 0;
            send_direct_payment(account_idx, next);
        } else {
            save_storage();
        }
    }
}

void sync_trustline(signed char account_idx, unsigned char *args) {
    unsigned long long limit;
    ntohn(&limit, args, 8, sizeof(long long));
    struct account *acc = &storage.accounts[account_idx];
    acc->creditlimit_in = limit;
    save_storage();
    account_send_8(account_idx, ACCOUNT_ACK_TRUSTLINE, limit);
}

void ack_trustline(signed char account_idx, unsigned char *args) {
    unsigned long long limit;
    ntohn(&limit, args, 8, sizeof(long long));

    struct account *acc = &storage.accounts[account_idx];

    if (!acc->ack_pending || acc->creditlimit != limit) {
        return;
    }

    acc->ack_pending = 0;
    save_storage();
}

void direct_transfer(signed char account_idx, unsigned char *args) {
    unsigned long long amount;
    ntohn(&amount, args, 8, sizeof(long long));
    unsigned long long counter;
    ntohn(&counter, args + 8, 8, sizeof(long long));
    unsigned char type = args[16];
    struct account *acc = &storage.accounts[account_idx];
    if (counter > acc->sync_in) {
        long long old_balance = acc->balance;
        acc->balance += (long long)amount;
        if (type && old_balance < 0) {
            unsigned long long out_amount = acc->balance > 0 ? (unsigned long long)-old_balance : amount;
            redistribute(account_idx, out_amount, acc->width);
        }
        acc->sync_in = counter;
        save_storage();
    }
    account_send_8(account_idx, ACCOUNT_ACK_TRANSFER, counter);
}

void ack_transfer(signed char account_idx, unsigned char *args) {
    unsigned long long counter;
    ntohn(&counter, args, 8, sizeof(long long));
    struct account *acc = &storage.accounts[account_idx];
    if (acc->direct_transfer > 0 && counter == acc->sync_out) {
        acc->balance -= (long long)acc->direct_transfer;
        acc->direct_transfer = 0;
        if (direct_payment_pending[account_idx] > 0) {
            unsigned long long amount = direct_payment_pending[account_idx];
            direct_payment_pending[account_idx] = 0;
            send_direct_payment(account_idx, amount);
        } else {
            save_storage();
        }
    }
}

void cpt_new_payment(signed char cpt_idx, unsigned char *args) {
    unsigned long long amount;
    unsigned long long fee;
    unsigned long long tax;
    ntohn(&amount, args,      8, sizeof(long long));
    ntohn(&fee,    args + 8,  8, sizeof(long long));
    ntohn(&tax,    args + 16, 8, sizeof(long long));

    if (num_paths >= MAX_CAPACITY - storage.num_payments || !validate_payment_overflow(amount, fee, tax) || amount == 0) {
        return;
    }

    unsigned char pay_id_buf[SHA256_DIGEST_SIZE];
    sha256(counterparts[cpt_idx].auth.secret_key, 32, pay_id_buf);

    if (get_pathfinding(pay_id_buf) != -1 || get_payment_idx(pay_id_buf) != -1) {
        return;
    }

    add_pathfinding(pay_id_buf, cpt_idx, amount, tax, fee, 0);

    forward_find_path((unsigned char)num_paths-1);
}

void cpt_find_path(signed char cpt_idx, unsigned char *args) {
    (void)args;
    unsigned char payment_id[32];
    sha256(counterparts[cpt_idx].auth.secret_key, 32, payment_id);
    signed char pf_idx = get_pathfinding(payment_id);
    if (pf_idx == -1) return;
    forward_find_path((unsigned char)pf_idx);
}

void cpt_commit_payment(signed char cpt_idx, unsigned char *args) {
    (void)args;
    unsigned char payment_id[32];
    sha256(counterparts[cpt_idx].auth.secret_key, 32, payment_id);

    signed char pf_idx = get_pathfinding(payment_id);
    if (pf_idx == -1) return;

    struct pathfinding *pf = &paths[pf_idx];

    if (!pf->in_or_out || !pf->commit) return;

    struct payment *payment = &storage.payments[storage.num_payments];
    storage.num_payments++;

    memcpy(&payment->base, &pf->base, (char*)&pf->base.time_value - (char*)&pf->base);
    payment->base.time_value = time(NULL);
    memcpy(payment->preimage, counterparts[cpt_idx].auth.secret_key, 32);

    payment->bits = 0;

    save_storage();

    paths[pf_idx] = paths[--num_paths];

    account_send_32_bytes(pf->base.outgoing, ACCOUNT_COMMIT_PAYMENT, payment_id);
}

void cpt_seal_payment(signed char cpt_idx, unsigned char *args) {
    (void)args;
    unsigned char payment_id[32];
    sha256(counterparts[cpt_idx].auth.secret_key, 32, payment_id);

    signed char pay_idx = get_payment_idx(payment_id);
    if (pay_idx == -1) return;

    struct payment *payment = &storage.payments[pay_idx];

    if (payment->status != COMMIT || payment->base.incoming != -1) return;

    unsigned long long ticker = penalty_ticker(payment);
    payment->commit_penalty = ticker;

    if (ticker >= payment->base.amount) {
        payment->status = CLEANUP;
        save_storage();
        account_send_32_bytes(payment->base.outgoing, ACCOUNT_CLEANUP_PAYMENT, payment_id);
        return;
    }

    payment->status = SEAL;
    save_storage();
    account_send_32_bytes(payment->base.outgoing, ACCOUNT_SEAL_PAYMENT, payment_id);
}

void cpt_start_payment(signed char cpt_idx, unsigned char *args) {
    unsigned char pay_id_buf[32];
    sha256(counterparts[cpt_idx].auth.secret_key, 32, pay_id_buf);
    signed char pf_idx = get_pathfinding(pay_id_buf);
    if (pf_idx == -1) return;
    cpt_send_new_payment(cpt_idx, pf_idx);
}

unsigned short (*user_handlers[])(unsigned char*) = {
    set_trust_index, add_account, remove_account, set_trustline, new_payment, 
    receive_payment, local_payment, get_trust_index, get_num_accounts,
    get_account, get_payment, get_receipt
};
void (*acc_handlers[])(signed char, unsigned char*) = {
    find_path, path_recurse, path_found, prepare_path, commit_payment, seal_payment,
    finalize_payment, cancel_payment, cleanup_payment, ack_preimage, sync_payment,
    ack_sync, verify_commit, ack_commit, direct_payment, ack_payment, sync_trustline, 
    ack_trustline, direct_transfer, ack_transfer
};
void (*cpt_handlers[])(signed char, unsigned char*) = {
    cpt_new_payment, cpt_start_payment, cpt_find_path, cpt_commit_payment, cpt_seal_payment
};

unsigned short user_args_len[] = {4, 98, 64, 72, 106, 98, 72, 0, 0, 1, 1, 1};
unsigned short acc_args_len[] = {59, 33, 33, 32, 32, 32, 32, 32, 32, 32, 40, 32, 32, 33, 24, 9, 8, 8, 17, 8};
unsigned short cpt_args_len[]  = {24, 0, 0, 0, 0};

void push_payment_instructions(void) {
    for (unsigned char i = 0; i < storage.num_payments; i++) {
        struct payment *p = &storage.payments[i];
        if (!p->ack_preimage) {
            if (p->status == FINALIZE && p->base.incoming != -1) {
                account_send_32_bytes(p->base.incoming, ACCOUNT_FINALIZE_PAYMENT, p->preimage);
            } else if (p->status == CANCEL && p->base.outgoing != -1) {
                account_send_32_bytes(p->base.outgoing, ACCOUNT_CANCEL_PAYMENT, p->preimage);
            }
        }

        if ((p->status & SYNCING_MASK) && p->base.outgoing != -1 && !p->ack_sync) {
            if ((p->status == CANCEL || p->status == CLEANUP) && !p->commit_out) {
                account_send_32_bytes(p->base.outgoing, ACCOUNT_VERIFY_COMMIT, p->base.identifier);
            } else {
                account_send_sync_payment(p->base.outgoing, p->base.identifier, p->finalize_out);
            }
        }
    }
}

void push_sync_account(void) {
    unsigned short temp_bitmap = storage.acc_bitmap;
    signed char idx = 0;
    
    while (temp_bitmap) {
        if (temp_bitmap & 1) {
            struct account *acc = &storage.accounts[idx];
            if (acc->ack_pending) {
                account_send_8(idx, ACCOUNT_SYNC_TRUSTLINE, acc->creditlimit);
            }            
            if (acc->direct_transfer > 0) {
                if (acc->type == 0) {
                    account_send_direct_payment(idx);
                } else {
                    send_buf[101] = ACCOUNT_DIRECT_TRANSFER;
                    htonn(&acc->direct_transfer, send_buf + 102, 8, sizeof(long long));
                    htonn(&acc->sync_out, send_buf + 110, 8, sizeof(long long));
                    send_buf[118] = acc->type - 1;
                    account_sign_and_send(idx, 1 + 17);
                }
            }
        }
        idx++;
        temp_bitmap >>= 1;
    }
}

void auto_cancel_routine(void) {
    long long now = time(NULL);
    for (unsigned char i = 0; i < storage.num_payments; i++) {
        struct payment *p = &storage.payments[i];
        if (p->status != COMMIT || !counterpart_in(p) || now - p->base.time_value <= TIMEOUT) continue;
        p->status = CANCEL;
        save_storage();
        account_send_32_bytes(p->base.outgoing, ACCOUNT_CANCEL_PAYMENT, p->preimage);
    }
}

void cleanup_payment_routine(void) {
    for (unsigned char i = 0; i < storage.num_payments; i++) {
        struct payment *p = &storage.payments[i];
        if ((p->status & SYNCING_MASK) || penalty_ticker(p) < p->base.amount) continue;
        if (p->status == COMMIT) {
            if (counterpart_in(p)) continue;
            p->commit_penalty = p->base.amount;
        }
        if (p->base.incoming != -1) {
            account_send_32_bytes(p->base.incoming, ACCOUNT_CLEANUP_PAYMENT, p->base.identifier);
        }
        if (p->base.outgoing != -1) {
            p->finalize_out = cleanup_out(p);
            account_send_32_bytes(p->base.outgoing, ACCOUNT_CLEANUP_PAYMENT, p->base.identifier);
        }
        p->status = CLEANUP;
        save_storage();
    }
}

void cleanup_buyer_timeouts(void) {
    if (cpt_bitmap) {
        long long now = time(NULL);
        for (unsigned char i = 0; i < num_paths; ) {
            struct pathfinding *pf = &paths[i];
            if (is_counterpart(&pf->base) && pf->in_or_out && pf->base.time_value < now) {
                remove_path_at(i, 1);
            } else {
                i++;
            }
        }
    }
}

void process_gas_buffers(void) {
    unsigned short temp_bitmap = storage.acc_bitmap;
    signed char idx = 0;
    
    while (temp_bitmap) {
        if (temp_bitmap & 1) {
            if (gas_buffer[idx] > 0) {
                struct account *acc = &storage.accounts[idx];
                if (acc->direct_transfer == 0) {
                    acc->sync_out++;
                    acc->direct_transfer = gas_buffer[idx];
                    acc->type = 1;
                    save_storage();
                    
                    send_buf[101] = ACCOUNT_DIRECT_TRANSFER;
                    htonn(&acc->direct_transfer, send_buf + 102, 8, sizeof(long long));
                    htonn(&acc->sync_out, send_buf + 110, 8, sizeof(long long));
                    send_buf[118] = 0;
                    account_sign_and_send(idx, 1 + 17);
                    
                    gas_buffer[idx] = 0;
                }
            }
        }
        idx++;
        temp_bitmap >>= 1;
    }
}

void push_new_session(void) {
    unsigned short temp_bitmap = storage.acc_bitmap;
    signed char idx = 0;
    
    while (temp_bitmap) {
        if (temp_bitmap & 1) {
            if (session_auth[idx].syncing) {
                new_session(idx);
                remove_sender(idx);
                struct account *acc = &storage.accounts[idx];
                remote_sign_and_send(ACCOUNT_NEW_SESSION, idx, &acc->remote, acc->secret_key, acc->counter, 0);
            }
        }
        idx++;
        temp_bitmap >>= 1;
    }
}

void ack_received(signed char sender_idx, unsigned char *data, unsigned char offset) {
    for (unsigned char i = 0; i < num_active_sends; i++) {
        struct send_rtx *t = &send_tasks[i];
        if (t->sender_idx == sender_idx) {
            if (memcmp(t->send_queue[t->head].payload + offset, data+offset, 4) == 0) {
                remove_send_task(i);
            }
            break;
        }
    }
}

signed char receive_tx(unsigned char session_type, signed char sender_idx, struct remote *remote, unsigned char *secret_key, unsigned int *counter, unsigned char *data, unsigned short len) {
    if (!verify_signature(data, len, secret_key)) {
        return -1;
    }
    if (data[0]&1) {
        ack_received(sender_idx, data, 97);
        return -1;
    }
    unsigned int counter_in;
    ntohn(&counter_in, data+97, 4, sizeof(int));
    if (counter_in < *counter) {
        return -1;
    }
    memcpy(send_buf+97, data+97, 4);
    build_tx_and_sign(session_type | 1, remote, secret_key, 0);
    struct sockaddr_in addr;
    create_sockaddr_in(remote->server_address, remote->port, &addr);
    sendto(sockfd, send_buf, TX_ENVELOPE, 0, (struct sockaddr *)&addr, sizeof(struct sockaddr_in));

    if (counter_in == *counter) {
        return -1;
    }
    *counter = counter_in;
    return 0;
}

#if 1
    #define DEBUG_PRINT_MESSAGE(addr, type, cmd) print_message(addr, type, cmd)
    const char* command_names[] = {
        "ACCOUNT_FIND_PATH", "ACCOUNT_PATH_RECURSE", "ACCOUNT_PATH_FOUND", "ACCOUNT_PREPARE_PATH",
        "ACCOUNT_COMMIT_PAYMENT", "ACCOUNT_SEAL_PAYMENT", "ACCOUNT_FINALIZE_PAYMENT", "ACCOUNT_CANCEL_PAYMENT",
        "ACCOUNT_CLEANUP_PAYMENT", "ACCOUNT_ACK_PREIMAGE", "ACCOUNT_SYNC_PAYMENT", "ACCOUNT_ACK_SYNC",
        "ACCOUNT_VERIFY_COMMIT", "ACCOUNT_ACK_COMMIT", "ACCOUNT_DIRECT_PAYMENT", "ACCOUNT_ACK_PAYMENT",
        "ACCOUNT_SYNC_TRUSTLINE", "ACCOUNT_ACK_TRUSTLINE", "ACCOUNT_DIRECT_TRANSFER", "ACCOUNT_ACK_TRANSFER",
        "COUNTERPART_NEW_PAYMENT", "COUNTERPART_START_PAYMENT", "COUNTERPART_FIND_PATH", 
        "COUNTERPART_COMMIT_PAYMENT", "COUNTERPART_SEAL_PAYMENT",
        "USER_SET_TRUST_INDEX", "USER_ADD_ACCOUNT", "USER_REMOVE_ACCOUNT", "USER_SET_TRUSTLINE", "USER_NEW_PAYMENT",
        "USER_RECEIVE_PAYMENT", "USER_LOCAL_PAYMENT", "USER_GET_TRUST_INDEX", "USER_GET_NUM_ACCOUNTS", "USER_GET_ACCOUNT",
        "USER_GET_PAYMENT", "USER_GET_RECEIPT"
    };
    void print_message(struct sockaddr_in *addr, unsigned char type, unsigned char cmd) {
        printf("%s:%d: %s (0x%02X)\n", inet_ntoa(addr->sin_addr), ntohs(addr->sin_port), 
               command_names[cmd + (type==4)*20 + (type==8)*25], cmd); fflush(stdout);
    }
#else
    #define DEBUG_PRINT_MESSAGE(addr, type, cmd) ((void)0)
#endif

unsigned char local;

void user_datagram(unsigned char *data, unsigned short len, struct sockaddr_in *addr) {
    unsigned char cmd;
    unsigned char *args;
    unsigned char envelope;
    if (!local) {
        if (len < REQ_ENVELOPE || !verify_signature(data, len, secret_block)) {
            return;
        }
        if (data[0]&1) {
            ack_received(USER_SENDER, data, 1);
            return;
        }
        unsigned int counter;
        ntohn(&counter, data+1, 4, sizeof(int));
        if (counter < storage.counter) {
            return;
        }

        send_buf[0] = 1;
        memcpy(send_buf+1, data+1, 4);
        sign_data(REQ_ENVELOPE-32, secret_block);
        sendto(sockfd, send_buf, REQ_ENVELOPE, 0, (struct sockaddr *)addr, sizeof(struct sockaddr_in));
        if (counter == storage.counter) {
            return;
        }
        storage.counter = counter;
        save_storage();
        cmd = data[5];
        if (cmd == USER_ADD_ACCOUNT || cmd == USER_NEW_PAYMENT || cmd == USER_RECEIVE_PAYMENT) {
            memcpy(secret_block + 32, data + 1, 4);
            sha256(secret_block, 36, one_time_key_buf);
            unsigned char *ciphertext = data+6+66;
            for (unsigned char i = 0; i < 32; i++) {
                ciphertext[i] ^= one_time_key_buf[i];
            }
        }
        args = data+6;
        envelope = REQ_ENVELOPE;
    } else {
        cmd = data[1];
        args = data+2;
        envelope = 1;
    }
    unsigned short out_len;
    if (cmd <= 11 && len-envelope >= 1 + user_args_len[cmd]) {
        if (cmd <= 6) DEBUG_PRINT_MESSAGE(addr, USER_SESSION, cmd);
        out_len = user_handlers[cmd](args);
    } else {
        out_len = error_message("Invalid instruction", 20);
    }
    if (!local) {
        send_buf[0] = 0;
        memcpy(send_buf+1, data+1, 4);
        sign_data(5+out_len, secret_block);
        enqueue_send(USER_SENDER, REQ_ENVELOPE+out_len, addr);
    } else {
        sendto(sockfd, send_buf+5, out_len, 0, (struct sockaddr *)addr, sizeof(struct sockaddr_in));
    }
}

void transport_datagram(unsigned char *data, unsigned short len, struct sockaddr_in *addr) {
    if (len<1) return;
    unsigned char session_type = data[0]&~1;
    if (session_type < 8) {
        if (len < TX_ENVELOPE) {
            return;
        }
        switch (session_type) {
        case ACCOUNT_NEW_SESSION: {
            signed char account_idx = lookup_account_idx(data+33);
            if (account_idx == -1) {
                return;
            }
            struct account *acc = &storage.accounts[account_idx];
            if (receive_tx(ACCOUNT_NEW_SESSION, account_idx, &acc->remote, acc->secret_key, &acc->counter, data, len) == -1) {
                return;
            }
            save_storage();
            set_session_auth(account_idx);
            remove_sender(account_idx);
            break;
        }
        case ACCOUNT_SESSION: {
            signed char account_idx = lookup_account_idx(data+33);
            if (account_idx == -1) {
                return;
            }
            struct auth *auth = &session_auth[account_idx].auth;
            if (receive_tx(ACCOUNT_SESSION, account_idx, &storage.accounts[account_idx].remote, auth->secret_key, &auth->counter_in, data, len) == -1) {
                return;
            }
            unsigned char cmd = data[101];
            if (cmd <= 19 && len-TX_ENVELOPE >= 1 + acc_args_len[cmd]) {
                DEBUG_PRINT_MESSAGE(addr, session_type, cmd);
                acc_handlers[cmd](account_idx, data+102);
            }
            break;
        }
        case COUNTERPART_SESSION: {
            signed char cpt_idx = lookup_cpt_idx(data+33, data+65);
            if (cpt_idx == -1) {
                return;
            }
            struct auth *auth = &counterparts[cpt_idx].auth;
            if (receive_tx(COUNTERPART_SESSION, (signed char)(CPT_SENDER_BASE + cpt_idx), &counterparts[cpt_idx].remote, auth->secret_key, &auth->counter_in, data, len) == -1) {
                return;
            }
            unsigned char cmd = data[101];
            if (cmd <= 4 && len - TX_ENVELOPE >= 1 + cpt_args_len[cmd]) {
                DEBUG_PRINT_MESSAGE(addr, session_type, cmd);
                cpt_handlers[cmd](cpt_idx, data+102);
            }
            break;
        }
        }
    } else {
        user_datagram(data, len, addr);
    }
}

#include <signal.h>

volatile sig_atomic_t running = 1;

void on_shutdown_signal(int sig) { running = 0; }

void run_loop(int in_sockfd) {
    unsigned int period = 0;
    struct timeval tv;
    tv.tv_usec = 0;
    unsigned char buf[MAX_DG];
    struct sockaddr_in src_addr;
    socklen_t addrlen;

    while (running) {
        time_t now = time(NULL);
        unsigned int current_period = (unsigned int)now / SCHEDULER_PERIOD;

        if (current_period > period) {
            period = current_period;
            push_payment_instructions();
            push_sync_account();
            cleanup_buyer_timeouts();
            if (current_period & 1) auto_cancel_routine();
            if (!(current_period & 3)) {
                cleanup_payment_routine();
                push_new_session();
            }
            if (!(current_period & 63)) process_gas_buffers();
        }
        next_timer = (current_period + 1) * SCHEDULER_PERIOD;

        retransmit(now);
        tv.tv_sec = next_timer - now;

        setsockopt(in_sockfd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
        addrlen = sizeof(struct sockaddr_in);
        ssize_t n = recvfrom(in_sockfd, buf, sizeof(buf), 0, (struct sockaddr *)&src_addr, &addrlen);

        if (n >= 0) {
            transport_datagram(buf, (unsigned short)n, &src_addr);
        }
    }
}

#include <sys/stat.h>

unsigned char nibble_to_byte(char c) {
    return (c >= '0' && c <= '9') ? c - '0' : c - 'a' + 10;
}

unsigned short parse_port(const char *str) {
    unsigned short port = 0;
    while (*str >= '0' && *str <= '9') {
        port = port * 10 + (*str++ - '0');
    }
    return port;
}

int main(int argc, char *argv[]) {
    if ((unsigned char)~0 != 0xFF) return 1;
    endianess = 0x0001;
    endianess = (*(char*)&endianess);

    if (argc >= 3) {
        strcpy(storage_path, argv[argc == 6 ? 5 : 2]);
    } else {
        struct stat sb;
        if (stat(STORAGE_DIR, &sb) != 0) {
            if (mkdir(STORAGE_DIR, 0755) != 0) {
                return 1;
            }
        }
        strcpy(storage_path, STORAGE_DIR "/" STORAGE_FILE);
    }

    if (argc >= 5) {
        string_copy(storage.username, argv[1], 32);
        strcpy(storage.server_address, argv[2]);
        storage.port = parse_port(argv[3]);
        char *key_str = argv[4];
        for (size_t i = 0; i < 64; i += 2) {
            storage.secret_key[i / 2] = (nibble_to_byte(key_str[i]) << 4) | nibble_to_byte(key_str[i + 1]);
        }
        save_storage();
        return 0;
    }

    if (load_storage() == -1) {
        return 1;
    }

    unsigned short port;
    if (argc < 4) {
        port = storage.port;
    } else {
        port = parse_port(argv[3]);
    }

    local = argc > 1 && argv[1][0] == '1';

    memcpy(secret_block, storage.secret_key, 32);

    struct sockaddr_in server_addr;
    int in_sockfd = socket(AF_INET, SOCK_DGRAM, 0);
    if (in_sockfd < 0) {
        return 1;
    }
    server_addr.sin_family = AF_INET;
    server_addr.sin_port = htons(port);
    if (local) {
        server_addr.sin_addr.s_addr = inet_addr("127.0.0.1");
        sockfd = socket(AF_INET, SOCK_DGRAM, 0);
        if (sockfd < 0) {
            return 1;
        }
    } else {
        server_addr.sin_addr.s_addr = INADDR_ANY;
        sockfd = in_sockfd;
    }
    if (bind(in_sockfd, (struct sockaddr *)&server_addr, sizeof(server_addr)) < 0) {
        return 1;
    }

    struct sigaction sa;
    sa.sa_handler = on_shutdown_signal;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = 0;
    sigaction(SIGINT, &sa, NULL);
    sigaction(SIGTERM, &sa, NULL);

    unsigned short temp_bitmap = storage.acc_bitmap;
    signed char i = 0;
    while (temp_bitmap) {
        if (temp_bitmap & 1) {
            struct account *acc = &storage.accounts[i];
            memcpy(session_auth[i].secret_block, acc->secret_key, 32);
            new_session(i);
            remote_sign_and_send(ACCOUNT_NEW_SESSION, i, &acc->remote, acc->secret_key, acc->counter, 0);
        }
        i++;
        temp_bitmap >>= 1;
    };

    run_loop(in_sockfd);

    return 0;
}
