# federation.py
import socket
import ssl
import struct
import hashlib

class Federation:
    def __init__(self, users, ports, global_secret, domain, send_udp_command, cert_path, cert_name, key_name, add_payment_request):
        self.users = users
        self.ports = ports
        self.global_secret = global_secret
        self.domain = domain
        self.send_udp_command = send_udp_command
        self.cert_path = cert_path
        self.cert_name = cert_name
        self.key_name = key_name
        self.add_payment_request = add_payment_request
        self.running = True
    
    def handle_connection(self, ssl_sock):
        # Read message type
        msg_type = ssl_sock.read(1)
        if not msg_type:
            return
        
        # ADD_ACCOUNT request
        if msg_type[0] == 0:
            data = ssl_sock.read(64)
            if len(data) != 64:
                return
                
            client_domain = data[:32].decode('utf-8', errors='ignore').rstrip('\0')
            client_hash = data[32:64]
            
            server_hash = hashlib.sha256(
                self.global_secret + client_domain.encode()
            ).digest()
            
            combined_xor = bytes(a ^ b for a, b in zip(client_hash, server_hash))
            combined = hashlib.sha256(combined_xor).digest()
            ssl_sock.send(combined)
            
        # PAYMENT request
        elif msg_type[0] == 1:
            data = ssl_sock.read(128)
            if len(data) != 128:
                return
                
            recipient = data[0:32].decode('utf-8', errors='ignore').rstrip('\0')
            sender = data[32:64].decode('utf-8', errors='ignore').rstrip('\0')
            sender_domain = data[64:96].decode('utf-8', errors='ignore').rstrip('\0')
            key = data[96:128]
            
            if recipient not in self.users:
                ssl_sock.send(b'PAYMENT_NACK')
                return
            
            if not self.add_payment_request(recipient, sender, sender_domain, key):
                ssl_sock.send(b'PAYMENT_NACK')
                return
                
            ssl_sock.send(b'PAYMENT_ACK')
    
    def run(self):
        context = ssl.create_default_context(ssl.Purpose.CLIENT_AUTH)
        context.load_cert_chain(
            str(self.cert_path / self.cert_name),
            str(self.cert_path / self.key_name)
        )
        
        server_sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        server_sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        server_sock.bind(('0.0.0.0', 8080))
        server_sock.listen(5)
        
        while self.running:
            client_sock, _ = server_sock.accept()
            client_sock.settimeout(5.0)
            try:
                ssl_sock = context.wrap_socket(client_sock, server_side=True)
                self.handle_connection(ssl_sock)
            except:
                pass
            finally:
                client_sock.close()