# Resilience Network

Web-of-trust money + basic income

## Requirements

* Python 3 with Flask
* `/usr/local/bin/resilience` binary
* SSL certificates (for federation)

## Setup

```bash
gcc -o resilience resilience.c
cp resilience /usr/local/bin/
pip install flask
DOMAIN=example.com python proxy.py
```

## Ports

- 3000: Web interface
- 8080: Federation (TLS)
- 2012: Resilience Network (UDP)
- 9000+: Proxy ↔ Resilience (internal, one per user)

## Usage

The Resilience network is trust-backed decentralized multi-hop payments (Ryan Fugger's Ripple from 2003) with a redistribution mechanism built on top. Users set credit limits with people they trust. Payments can route through this trust network - if Bob trusts Alice and Carol trusts Bob, Alice can pay Carol through Bob. A configurable tax (trust index) on payments gets redistributed through the trust network to users with negative balances as basic income, flowing through the network of IOUs.

## Federation

Keys are negotiated automatically. Requires SSL certificates.

## API

- `/api/auth/*` - Authentication
- `/api/user/*` - User operations  
- `/api/federated/*` - Cross-server operations
- `/api/admin/*` - Admin only

## Technical Details

### Architecture

Each user gets their own `resilience` process managing payments and redistribution. The proxy handles multi-user management, federation protocol, and web interface.

### UDP Commands (Proxy → Resilience)

Each user's resilience process listens on its own UDP port starting from 9000. The proxy forwards incoming P2P traffic (port 2012) to the appropriate user's process.

| Cmd | Name | Arguments | Returns |
|-----|------|-----------|---------|
| 0 | SET_TRUST_INDEX | float[4] | status[1] message[N] |
| 1 | ADD_ACCOUNT | username[32] server[32] port[2] key[32] | status[1] message[N] |
| 2 | REMOVE_ACCOUNT | username[32] server[32] | status[1] message[N] |
| 3 | SET_TRUSTLINE | username[32] server[32] amount[8] | status[1] message[N] |
| 4 | NEW_PAYMENT | username[32] server[32] port[2] key[32] amount[8] | status[1] message[N] |
| 5 | RECEIVE_PAYMENT | username[32] server[32] port[2] key[32] | status[1] message[N] |
| 6 | LOCAL_PAYMENT | username[32] server[32] amount[8] | status[1] message[N] |
| 7 | GET_TRUST_INDEX | - | status[1] float[4] |
| 8 | GET_NUM_ACCOUNTS | - | status[1] count[1] |
| 9 | GET_ACCOUNT | index[1] | status[1] username[32] server[32] balance[8] limit_out[8] limit_in[8] |
| 10 | GET_PAYMENT | index[1] | status[1] state[1] id[32] username[32] server[32] amount[8] in_or_out[1] time_val[8] |
| 11 | GET_RECEIPT | index[1] | status[1] type[1] id[32] username[32] server[32] amount[8] timestamp[8] status[1] |

### Federation Protocol

Automatic key negotiation between servers using TLS on port 8080. When adding `username@server.com`, keys are derived from server secrets and negotiated automatically. Payment requests check recipient capacity before proceeding.

#### POST /api/federated/add_account
When adding a federated account:
1. Client connects to server via TLS on port 8080
2. Client sends: domain[32] + client_hash (SHA256 of global_secret + server_domain)
3. Server responds with combined hash
4. Client sends ADD_ACCOUNT command to resilience binary with the combined key

#### POST /api/federated/payment
Federated payments:
1. Sender generates random key for this payment
2. Sender connects to recipient's server and sends: recipient[32] + sender[32] + sender_domain[32] + key[32]
3. Server adds to payment_requests queue (max 16 pending)
4. Server responds with PAYMENT_ACK or PAYMENT_NACK
5. If ACK, sender proceeds with payment using the generated key
6. Recipient must manually accept the payment request through web UI
7. Upon acceptance, recipient sends RECEIVE_PAYMENT command to resilience binary with the key.

### API Endpoints (Port 3000)

#### Authentication
```
POST /api/auth/login                 {username, password}
POST /api/auth/logout    
GET  /api/auth/whoami
POST /api/auth/change_password       {old_password, new_password}
```

#### Admin (requires admin role)
```
POST /api/admin/create_user          {user_id, password}
POST /api/admin/delete_user          {user_id}
POST /api/admin/reset_password       {user_id, new_password}
GET  /api/admin/list_users
POST /api/admin/create_invite
```

#### User (requires auth)
```
GET  /api/user/accounts
GET  /api/user/receipts
GET  /api/user/trust_index
POST /api/user/trust_index           {value}
POST /api/user/add_account           {username, server, port, key}
POST /api/user/remove_account        {username, server}
POST /api/user/set_trustline         {username, server, amount}
POST /api/user/payment               {username, server, port, key, amount}
POST /api/user/receive_payment       {username, server, port, key}
POST /api/user/local_payment         {username, server, amount}
```

#### Federated (requires auth)
```
POST /api/federated/add_account      {username, server}
POST /api/federated/payment          {username, server, amount}
```

#### Invite
```
POST /api/invite/register            {invite, username, password}
```

# Resilience Binary Protocol Specification

## Message Format

### Session Types
```
0x00 - ACCOUNT_SESSION (trusted peers)
0x02 - COUNTERPART_SESSION (payment endpoints)
0x04 - USER_SESSION (client commands)
```

### Account Message Envelope
```
[1] session_type
[32] recipient_username
[32] sender_username  
[32] sender_server
[4] counter
[1] command
[N] arguments
[32] signature
```

### User Message Envelope
```
[1] session_type (0x04)
[4] counter
[1] command
[N] arguments
[32] signature
```

## Instructions

### Account Instructions
```
0x00 FIND_PATH          [32]id [8]amount [8]tax [1]in_or_out [8]fee [1]hops [1]gas
0x01 PATH_RECURSE       [32]id [1]depth
0x02 PATH_FOUND         [32]id [1]hops
0x03 PREPARE_PATH       [32]id
0x04 COMMIT_PAYMENT     [32]id
0x05 SEAL_PAYMENT       [32]id
0x06 FINALIZE_PAYMENT   [32]preimage
0x07 CANCEL_PAYMENT     [32]preimage
0x08 CLEANUP_PAYMENT    [32]id
0x09 ACK_PREIMAGE       [32]id
0x0A SYNC_PAYMENT       [32]id [8]amount
0x0B ACK_SYNC           [32]id
0x0C VERIFY_COMMIT      [32]id
0x0D ACK_COMMIT         [32]id [1]status
0x0E DIRECT_PAYMENT     [8]amount [8]tax [8]counter
0x0F ACK_PAYMENT        [8]counter [1]status
0x10 SYNC_TRUSTLINE     [8]limit
0x11 ACK_TRUSTLINE      [8]limit
0x12 DIRECT_TRANSFER    [8]amount [8]counter [1]type
0x13 ACK_TRANSFER       [8]counter
```

### Counterpart Instructions
```
0x00 NEW_PAYMENT        [8]amount [8]fee [8]tax
0x01 START_PAYMENT      
0x02 FIND_PATH         
0x03 COMMIT_PAYMENT    
0x04 SEAL_PAYMENT      
```

## Payment Flow

```
COMMIT ─→ SEAL ─→ FINALIZE (preimage)
   ↓   ↘   ↓
CANCEL  CLEANUP
(preimage)
```

## Secure UDP+Retransmission (UDPR)

UDP with automatic retransmission and cryptographic authentication.

### Message Format
```
[type:1][...][counter:4][...][signature:32]
```

- **Type**: 0=data, 1=ACK
- **Signature**: `sha256(msg+key)` with shared secret

Temporary session keys derived from base key + session counter.
